<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Shopify\ShopifyOrder;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\DB;

class SalesChannelAndSkuFulfillmentsMismatch extends Integrity implements Identifier, Remedy
{
    private Collection $records;

    private function getMismatchedLinesQuery(): Builder
    {
        $sql = <<<'SQL'
            CREATE TEMPORARY TABLE temporary_shopify_fulfillment_line_items (
                `id` bigint(20) unsigned NOT NULL AUTO_INCREMENT,
                `line_item_id` varchar(255),
                `fulfillment_id` bigint(20) unsigned,
                `shopify_sales_order_id` bigint(20) unsigned,
                `sales_order_name` varchar(255),
                `quantity` int(11),
                `fulfillment_status` varchar(255),
                `json_object` mediumtext,
                PRIMARY KEY (`id`), INDEX(`line_item_id`), INDEX (`quantity`)
            ) ENGINE=InnoDB DEFAULT COLLATE utf8mb4_unicode_ci  
        SQL;

        DB::statement($sql);

        ShopifyOrder::query()
            ->select(['id', 'name', 'fulfillments'])
            ->where('fulfillments', '!=', '[]')
            ->whereNotNull('fulfillments')
            ->chunk(10000, function (Collection $orders) {
                $orders->each(function (ShopifyOrder $order) {
                    $fulfillments = collect($order->fulfillments)->where('status', 'success')->all();
                    foreach ($fulfillments as $fulfillment) {
                        $this->insertFulfillmentIntoTempTable($order, $fulfillment);
                    }
                });
            });

        return SalesOrderLine::query()
            ->selectRaw('sol.id, SUM(temp_qty) as shopifyFulfilled, SUM(sofl_qty) as skuFulfilled')
            ->from('sales_order_lines', 'sol')
            ->join('sales_orders as so', 'so.id', 'sol.sales_order_id')
            ->joinSub(function ($query) {
                $query->selectRaw('sol.id as sol_id, sum(sofl.quantity) as sofl_qty')
                    ->from('sales_order_lines', 'sol')
                    ->join('sales_orders as so', 'so.id', 'sol.sales_order_id')
                    ->join('sales_order_fulfillment_lines as sofl', 'sofl.sales_order_line_id', 'sol.id')
                    ->join('sales_order_fulfillments as sof', 'sof.id', 'sofl.sales_order_fulfillment_id')
                    ->where('so.fulfillment_status', '!=', 'over_fulfilled')
                    ->where('sof.status', SalesOrderFulfillment::STATUS_FULFILLED)
                    ->groupBy('sol.id');
            }, 'sofl', 'sofl.sol_id', 'sol.id')
            ->joinSub(function ($query) {
                $query->selectRaw('sol.id as sol_id, sum(temp.quantity) as temp_qty')
                    ->from('sales_order_lines', 'sol')
                    ->join('sales_orders as so', 'so.id', 'sol.sales_order_id')
                    ->join('temporary_shopify_fulfillment_line_items as temp', 'temp.line_item_id', 'sol.sales_channel_line_id')
                    ->where('temp.fulfillment_status', 'success')
                    ->where('so.fulfillment_status', '!=', 'over_fulfilled')
                    ->groupBy('sol.id');
            }, 'temp', 'temp.sol_id', 'sol.id')
            ->where('sol.no_audit_trail', 0)
            ->where('sol.is_product', 1)
            ->where('so.fulfillment_status', '!=', 'over_fulfilled')
            ->groupBy('sol.id')
            ->havingRaw('shopifyFulfilled != skuFulfilled');
    }

    private function insertFulfillmentIntoTempTable(ShopifyOrder $order, array $fulfillment)
    {
        foreach ($fulfillment['line_items'] as $fulfillmentLine) {
            $fulfillmentLineId = $fulfillmentLine['id'];
            $fulfillmentId = $fulfillment['id'];
            $fulfillmentLineQty = $fulfillmentLine['quantity'];
            $fulfillmentStatus = $fulfillment['status'];
            $sql = "INSERT INTO temporary_shopify_fulfillment_line_items VALUES(NULL, $fulfillmentLineId, $fulfillmentId, $order->id, '$order->name', $fulfillmentLineQty, '$fulfillmentStatus', '".str_replace("'", "\'", json_encode($fulfillmentLine))."')";
            DB::statement($sql);
        }
    }

    public function identify(): void
    {
        $query = $this->getMismatchedLinesQuery();

        // We cache the records on the object so we can
        // use it in examples (if needed). Note that
        // we can't run the query multiple times in the
        // same session since we're using temporary tables.
        $this->records = $query->get();

        $total = $this->records->count();
        $this->addMessage('There are '.$total.' sales order lines with fulfillment quantity out of sync with Shopify', $total);
    }

    public function examples(): void
    {
        if ($this->records->isEmpty()) {
            return;
        }

        $this->printMessage("EXAMPLES - SC & SKU Fulfillments Mismatch:\n");
        foreach ($this->records->take(5) as $result) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = SalesOrderLine::with([])->find($result->id);
            $salesOrder = $salesOrderLine->salesOrder;
            $product = $salesOrderLine->product;

            $this->printMessage("$salesOrder->sales_order_number SKU: $product->sku (ID: $salesOrderLine->id) Shopify fulfilled ($result->shopifyFulfilled) != sku.io fulfilled ($result->skuFulfilled)");
        }
    }

    public function remedy(): void
    {
    }

    public function description(): string
    {
        return 'Fix disparity between sales channel and sku fulfillment quantities';
    }
}
