<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use Illuminate\Console\Command;

class Remedy extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:inventory:integrity:remedy
                                {--debug : Debug mode }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Remedy inventory integrity issues';

    protected array $remedies;

    public function __construct()
    {
        parent::__construct();

        $this->remedies = [
            new InvalidFulfilledQuantityCache($this),
            new MismatchedSalesOrderLineAndMovementQuantities($this),
            new MissingReservations($this),
            new NoAuditTrailLinesWithMovements($this),
            new OrphanedReservations($this),
            new InvalidFifoLayerFulfilledQuantityCache($this),
            new PreAuditTrailReservations($this),
            new OutOfSyncInventoryMovementReferences($this),
            /*
             * This identifier is disabled because we can't reliably detect the issue.  Also a future remedy may be to just mark
             * as out of sync.  Note example BG110499 which has 2 separate fulfillments and is producing a false positive.  Also
             * note that Non product lines being fulfilled may also produce a false positive for being out of sync.
             */
            //new SalesChannelAndSkuFulfillmentsMismatch($this),
        ];
    }

    public function handle(): int
    {
        $headers = ['No.', 'Remedy'];

        $data = array_map(function ($id, Identifier $remedy) {
            return [$id, $remedy->description()];
        }, array_keys($this->remedies), $this->remedies);

        $this->table($headers, $data);

        $remedyIndexInput = $this->ask('Enter numbers of remedies to run (comma separated) or "all" to process all:');

        if (isset($remedyIndexInput) && $remedyIndexInput !== 'all') {
            $remediesToRun = array_map('intval', explode(',', str_replace(' ', '', $remedyIndexInput)));
        } else {
            $remediesToRun = array_keys($this->remedies);
        }

        foreach ($this->remedies as $index => $remedy) {
            if (! in_array($index, $remediesToRun)) {
                continue;
            }

            $this->warn('Fixing...');
            $remedy->remedy();
            $this->info('DONE.');
        }

        return 0;
    }
}
