<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\Integration;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class PreAuditTrailReservations extends Integrity implements Identifier, Remedy
{
    private function preAuditTrailReservationsQuery(): Builder
    {
        $systemTimezone = Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, 'UTC');
        $systemInventoryStartDate = Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE, '1970-01-01 00:00:00'), $systemTimezone)->toDateTimeString();

        return SalesOrderLine::query()
            ->from('sales_order_lines', 'sol')
            ->selectRaw('sol.id')
            ->join('inventory_movements as im', 'im.link_id', 'sol.id')
            ->join('sales_orders as so', 'so.id', 'sol.sales_order_id')
            ->join('sales_channels as sc', 'sc.id', 'so.sales_channel_id')
            ->join('integration_instances as ii', 'ii.id', 'sc.integration_instance_id')
            ->where('im.link_type', SalesOrderLine::class)
            ->where('im.inventory_movement_date', '<', $systemInventoryStartDate)
            ->groupBy('sol.id');
    }

    public function identify(): void
    {
        $total = $this->preAuditTrailReservationsQuery()->count();
        $this->addMessage('There are '.$total.' sales order lines with reservations pre audit trail date', $total);
    }

    public function examples(): void
    {
        $query = $this->preAuditTrailReservationsQuery();
        if ($query->count() == 0) {
            return;
        }
        $this->printMessage("EXAMPLES: Pre-audit trail lines with reservations\n");
        foreach ($query->take(5)->get() as $result) {
            /** @var SalesOrderLine $orderLine */
            $orderLine = SalesOrderLine::with(['product', 'salesOrder'])->findOrFail($result->id);
            $this->printMessage("{$orderLine->salesOrder->sales_order_number}, SKU: {$orderLine->product->sku}");
        }
    }

    public function remedy(): void
    {
        $preAuditTrailReservations = $this->preAuditTrailReservationsQuery()->get();
        $this->console->info('There are '.$preAuditTrailReservations->count().' sales order lines with reservations pre audit trail date');
        $this->console->confirm('Would you like to continue?');
        foreach ($preAuditTrailReservations as $result) {
            DB::beginTransaction();
            try {
                /** @var SalesOrderLine $salesOrderLine */
                $salesOrderLine = SalesOrderLine::query()->findOrFail($result->id);
                $product = $salesOrderLine->product;
                $salesOrder = $salesOrderLine->salesOrder;

                $this->console->info('Sales Order '.$salesOrder->sales_order_number.'('.$salesOrder->order_date.'), SKU: '.$product->sku.' (Qty: '.$salesOrderLine->quantity.') Line ID ('.$salesOrderLine->id.')');

                $movements = $salesOrderLine->inventoryMovements->where('type', 'sale')->where('link_type', SalesOrderLine::class)->groupBy('layer_id');

                foreach ($movements as $layer_movements) {
                    foreach ($layer_movements as $movement) {
                        $this->console->info("\t".$movement->quantity." ($movement->inventory_status) belonging to layer $movement->layer_id ($movement->layer_type) ($movement->inventory_movement_date)");
                    }
                }

                if (! $salesOrderLine->fulfilled) {
                    $this->console->info('Sales order line is not fulfilled, so deleting reservations for pre audit trail order');

                    if ($this->debugging() && ! $this->console->confirm('Would you like to delete these movements?')) {
                        echo 'OK, skipping'."\n";

                        continue;
                    }

                    $this->deleteReservationsForLine($salesOrderLine);
                } else {
                    $this->console->info('Sales order line is fulfilled, using fulfillment date to determine whether to delete movements or update dates of reservations to match fulfillment');
                    $this->handleDatesBasedOnFulfillments($salesOrderLine);
                }

                DB::commit();
            } catch (Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            }
        }
    }

    private function deleteReservationsForLine(SalesOrderLine $salesOrderLine)
    {
        $this->console->warn("Deleting reservations and movements for SOL: $salesOrderLine->id");
        /** @var InventoryMovement $movement */
        foreach ($salesOrderLine->inventoryMovements as $movement) {
            $movement->delete();
            $movement->reverseLayer();
        }
        (new UpdateProductsInventoryAndAvgCost([$salesOrderLine->product_id]))->handle();
    }

    private function handleDatesBasedOnFulfillments(SalesOrderLine $salesOrderLine)
    {
        // For fulfillments pre-audit trail, we delete the fulfillment and movements for the
        // sales order. However, for post-audit-trail fulfillments, we update the inventory
        // movement dates for reservations to be the audit-trail start date for the sales channel.
        $inventoryStartDate = $this->getAuditTrailStartDateBySalesOrder($salesOrderLine->salesOrder);
        /** @var SalesOrderFulfillment $fulfillment */
        $fulfillment = $salesOrderLine->salesOrderFulfillmentLines->first()?->salesOrderFulfillment;
        if (! $fulfillment || $fulfillment->fulfilled_at->lessThan($inventoryStartDate)) {
            if ($this->debugging() && ! $this->console->confirm('Would you like to delete these movements?')) {
                echo 'OK, skipping'."\n";

                return;
            }
            // There is no post-audit trail fulfillment on the line, we delete the reservations.
            $this->deleteReservationsForLine($salesOrderLine);
            // We also delete any fulfillments
            $fulfillment?->delete(true);

            return;
        }

        // We reset the movement dates of the reservation to the inventory start date
        $this->console->warn('Updating reservation movements to audit trail start date...');
        $salesOrderLine->inventoryMovements()
            ->update([
                'inventory_movement_date' => $inventoryStartDate,
            ]);
    }

    private function getAuditTrailStartDateBySalesOrder(SalesOrder $salesOrder): Carbon
    {
        return ($salesOrder->salesChannel->integration->name === Integration::NAME_SKU_IO ||
            ! $salesOrder->salesChannel->integrationInstance->audit_trail_start_date) ?
            Helpers::setting(Setting::KEY_INVENTORY_START_DATE, '1970-01-01 00:00:00') :
            $salesOrder->salesChannel->integrationInstance->audit_trail_start_date;
    }

    public function description(): string
    {
        return 'Fix pre audit trail lines with reservations';
    }
}
