<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Helpers;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\InventoryMovement;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Carbon\Carbon;
use DB;
use Exception;
use Illuminate\Database\Eloquent\Builder;

class NoAuditTrailLinesWithMovements extends Integrity implements Identifier, Remedy
{
    private function SalesOrderLinesWithPreStartDateMovementsQuery(): Builder
    {
        return SalesOrderLine::with(['product', 'salesOrder'])
            ->select('sales_order_lines.id', 'sales_order_lines.sales_order_id', 'sales_order_lines.product_id')
            ->join('inventory_movements', 'inventory_movements.link_id', 'sales_order_lines.id')
            ->where('inventory_movements.inventory_movement_date', '<', Helpers::setting(Setting::KEY_INVENTORY_START_DATE))
            ->where('inventory_movements.link_type', SalesOrderLine::class)
            ->groupBy('sales_order_lines.id', 'sales_order_lines.sales_order_id', 'sales_order_lines.product_id');
    }

    private function SalesOrderFulfilmentLinesWithPreStartDateMovementQuery(): Builder
    {
        return SalesOrderFulfillmentLine::with(['salesOrderLine', 'salesOrderFulfillment'])
            ->select('sales_order_fulfillment_lines.*')
            ->join('inventory_movements', 'inventory_movements.link_id', 'sales_order_fulfillment_lines.id')
            ->where('inventory_movements.inventory_movement_date', '<', Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE)))
            ->where('inventory_movements.link_type', SalesOrderFulfillmentLine::class);
    }

    public function identify(): void
    {
        /*
         * Sales order lines no audit trail with movements
         */
        $total = $this->SalesOrderLinesWithPreStartDateMovementsQuery()->count();
        $this->addMessage('There are '.$total.' no audit trail sales order lines with inventory movements', $total);
    }

    public function examples(): void
    {
        $query = $this->SalesOrderLinesWithPreStartDateMovementsQuery();
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES: No Audit Trail lines with Movements\n");
        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("{$result->salesOrder->sales_order_number}, (SKU: {$result->product->sku})");
        }
    }

    /**
     * @throws Exception
     * @throws \Throwable
     */
    public function remedy(): void
    {
        $query = $this->SalesOrderLinesWithPreStartDateMovementsQuery();

        /** @var SalesOrderLine $salesOrderLine */
        foreach ($query->get() as $salesOrderLine) {
            $this->console->info($salesOrderLine->product->sku.' on '.$salesOrderLine->salesOrder->sales_order_number.' ('.$salesOrderLine->id.')');

            $salesOrderLine->inventoryMovements->each(function (InventoryMovement $movement) {
                $this->console->info('Moving inventory movement to start date of inventory, movement id: '.$movement->id);
                $movement->inventory_movement_date = Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE));
                $movement->save();
            });
        }

        $query = $this->SalesOrderFulfilmentLinesWithPreStartDateMovementQuery();

        $this->console->info($query->count().' fulfillment lines with pre start date movements');

        $productIds = [];
        /** @var SalesOrderFulfillmentLine $salesOrderFulfillmentLine */
        foreach ($query->get() as $salesOrderFulfillmentLine) {
            $salesOrderLine = $salesOrderFulfillmentLine->salesOrderLine;
            $this->console->info("Fulfillment line id $salesOrderFulfillmentLine->id for {$salesOrderLine->salesOrder->sales_order_number} ({$salesOrderLine->product->sku} belongs to fulfillment with date of ".$salesOrderFulfillmentLine->salesOrderFulfillment->fulfilled_at);
            if ($salesOrderFulfillmentLine->salesOrderFulfillment->created_at->lt(Carbon::now()->subDays(2))) {
                $this->console->info('Fulfillment is not recent, so changing fulfillment date movements to start date of inventory');
                $salesOrderFulfillmentLine->inventoryMovements->each(function (InventoryMovement $movement) {
                    $this->console->info('Moving inventory movement to start date of inventory, movement id: '.$movement->id);
                    $movement->inventory_movement_date = Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE));
                    $movement->save();
                });

                continue;
            }
            if (! $this->debugging() || ($this->debugging() && ! $this->console->confirm('Would you like to delete this fulfillment?'))) {
                continue;
            } else {
                $this->console->info("Deleting fulfillment for no audit trail line: {$salesOrderLine->salesOrder->sales_order_number} ({$salesOrderLine->product->sku})");
                DB::transaction(function () use ($salesOrderFulfillmentLine, $salesOrderLine, &$productIds) {
                    $salesOrderFulfillmentLine->salesOrderFulfillment->salesOrderFulfillmentLines->each(function (SalesOrderFulfillmentLine $salesOrderFulfillmentLine) {
                        $salesOrderFulfillmentLine->salesOrderLine->externally_fulfilled_quantity = $salesOrderFulfillmentLine->quantity;
                        $salesOrderFulfillmentLine->salesOrderLine->warehouse_id = null;
                        $salesOrderFulfillmentLine->salesOrderLine->save();
                    });
                    $salesOrderFulfillmentLine->salesOrderFulfillment->delete(true);
                    $productIds[] = $salesOrderLine->product_id;
                });
            }
        }

        if (count($productIds) > 0) {
            (new UpdateProductsInventoryAndAvgCost($productIds))->handle();
        }
    }

    public function description(): string
    {
        return 'Fix no audit trail sales order lines with movements';
    }
}
