<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Actions\InventoryManagement\GetFifoLayersForQuantity;
use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Data\FifoLayerQuantityData;
use App\Exceptions\InsufficientStockException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillmentLine;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Models\Warehouse;
use App\Repositories\FifoLayerRepository;
use App\Repositories\InventoryMovementRepository;
use App\Services\InventoryManagement\InventoryManager;
use Exception;
use Illuminate\Database\Query\Builder;
use Illuminate\Support\Facades\DB;
use Throwable;

class MismatchedSalesOrderLineAndMovementQuantities extends Integrity implements Identifier, Remedy
{
    public function description(): string
    {
        return 'Fix mismatched sales order lines and inventory movements quantities';
    }

    private function unmatchedMovementsAndLineQuantitiesQuery(): Builder
    {
        return DB::table('sol')
            ->select(DB::raw('sol.id, sol_qty, reserved_qty, active_qty'))
            ->withExpression('sol', function ($query) {
                $query->selectRaw('id, product_id, sales_order_id, sum(quantity) sol_qty')
                    ->from('sales_order_lines')
                    ->whereNotNull('product_id')
                    ->groupBy('id', 'product_id', 'sales_order_id');
            })
            ->withExpression('im_reserved', function ($query) {
                $query->selectRaw('link_id as r_solid, sum(quantity) reserved_qty')
                    ->from('inventory_movements')
                    ->where('link_type', SalesOrderLine::class)
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->where('quantity', '>', 0)
                    ->groupBy('link_id');
            })
            ->withExpression('im_active', function ($query) {
                $query->selectRaw('link_id as a_solid, sum(abs(quantity)) active_qty')
                    ->from('inventory_movements')
                    ->where('link_type', SalesOrderLine::class)
                    ->where('inventory_status', 'active')
                    ->groupBy('link_id');
            })
            ->join('im_reserved', 'sol.id', 'im_reserved.r_solid')
            ->join('im_active', 'sol.id', 'im_active.a_solid')
            ->whereColumn('sol_qty', '!=', 'reserved_qty')
            ->orWhereColumn('sol_qty', '!=', 'active_qty');
    }

    private function nonZeroSumReservationsQuery(array $relations = []): \Illuminate\Database\Eloquent\Builder
    {
        $totalQuantitySubquery = DB::table('inventory_movements')
            ->selectRaw('link_id, sum(quantity) as total_quantity')
            ->where('link_type', SalesOrderLine::class)
            ->groupBy('link_id');

        return SalesOrderLine::with($relations)
            ->select('sales_order_lines.*')
            ->join('sales_orders', 'sales_orders.id', 'sales_order_lines.sales_order_id')
            ->join('products', 'products.id', 'sales_order_lines.product_id')
            ->leftJoinSub($totalQuantitySubquery, 'total_quantity', function ($join) {
                $join->on('sales_order_lines.id', '=', 'total_quantity.link_id');
            })
            ->where('sales_orders.order_status', '!=', SalesOrder::STATUS_DRAFT)
            ->where('total_quantity.total_quantity', '!=', 0);
    }

    private function movementsForDraftOrdersQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return InventoryMovement::query()
            ->selectRaw('im.*')
            ->from((new InventoryMovement)->getTable(), 'im')
            ->join('sales_order_lines as sol', function ($join) {
                $join->on('im.link_id', 'sol.id')
                    ->where('im.link_type', SalesOrderLine::class);
            })
            ->join('sales_orders as so', 'sol.sales_order_id', 'so.id')
            ->where('so.order_status', SalesOrder::STATUS_DRAFT)
            ->union(function ($query) {
                $query->selectRaw('im.*')
                    ->from((new InventoryMovement)->getTable(), 'im')
                    ->join('sales_order_fulfillment_lines as sofl', function ($join) {
                        $join->on('im.link_id', 'sofl.id')
                            ->where('im.link_type', SalesOrderFulfillmentLine::class);
                    })
                    ->join('sales_order_lines as sol', 'sol.id', 'sofl.sales_order_line_id')
                    ->join('sales_orders as so', 'sol.sales_order_id', 'so.id')
                    ->where('so.order_status', SalesOrder::STATUS_DRAFT);
            });
    }

    public function fulfillmentLinesWithoutMovementsQuery(): \Illuminate\Database\Eloquent\Builder
    {
        return SalesOrderFulfillmentLine::query()
            ->selectRaw('sales_order_fulfillment_lines.*, sales_order_fulfillment_lines.quantity fulfilled_qty, 
        (SELECT abs(coalesce(sum(im.quantity), 0)) 
         FROM `inventory_movements` as `im` 
         WHERE `im`.`link_id` = `sales_order_fulfillment_lines`.`id` 
         AND `im`.`link_type` = ?) as movement_qty', [SalesOrderFulfillmentLine::class])
            ->join('sales_order_lines', 'sales_order_lines.id', '=',
                'sales_order_fulfillment_lines.sales_order_line_id')
            ->join('sales_orders', 'sales_orders.id', '=', 'sales_order_lines.sales_order_id')
            ->join('warehouses', function ($join) {
                $join->on('warehouses.id', '=', 'sales_order_lines.warehouse_id')
                    ->where('warehouses.type', '!=', Warehouse::TYPE_SUPPLIER);
            })
            ->where('fulfillment_status', '!=', SalesOrder::FULFILLMENT_STATUS_OVER_FULFILLED)
            ->havingRaw('fulfilled_qty != movement_qty');
    }

    public function identify(): void
    {
        $total = $this->unmatchedMovementsAndLineQuantitiesQuery()->count();
        $this->addMessage('There are '.$total.' sales order lines with differing movement and line quantities.', $total);

        $results = InventoryMovement::with([])->where('quantity', 0)->get();

        $total = count($results);
        $this->addMessage('There are '.count($results).' inventory movements with 0 qty', $total);

        // Draft orders with movements
        $total = $this->movementsForDraftOrdersQuery()->count();
        $this->addMessage('There are '.$total.' inventory movements belonging to draft orders', $total);

        $total = $this->nonZeroSumReservationsQuery()->count();
        $this->addMessage('There are '.$total.' sales order lines with non-zero sum for reservation movements', $total);

        $total = $this->fulfillmentLinesWithoutMovementsQuery()->count();
        $this->addMessage('There are '.$total.' fulfillment lines without inventory movements', $total);
    }

    public function examples(): void
    {
        $this->unmatchedMovementsAndLineQuantitiesExamples();
        $this->zeroUnitsMovementsExamples();
        $this->movementsForDraftOrdersExamples();
        $this->nonzeroSumReservationsExamples();
        $this->fulfillmentLinesWithoutMovementsExamples();
    }

    private function fulfillmentLinesWithoutMovementsExamples()
    {
        $query = $this->fulfillmentLinesWithoutMovementsQuery()->with(['salesOrderLine.salesOrder', 'salesOrderLine.product']);
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES - Fulfillment lines without/insufficient movements (quantities)\n");
        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("{$result->salesOrderLine->salesOrder->sales_order_number}, SKU: {$result->salesOrderLine->product->sku}, Fulfilled qty: $result->fulfilled_qty, Movement qty: $result->movement_qty");
        }
    }

    private function nonzeroSumReservationsExamples()
    {
        $query = $this->nonZeroSumReservationsQuery()->with(['product', 'salesOrder']);
        if ($query->count() == 0) {
            return;
        }

        $this->printMessage("EXAMPLES - Non-zero sum Reservations\n");

        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("{$result->salesOrder->sales_order_number} (SKU: {$result->product->sku})");
        }
    }

    private function movementsForDraftOrdersExamples()
    {
        $query = $this->movementsForDraftOrdersQuery()->with(['product', 'link.salesOrder']);
        if ($query->count() > 0) {
            return;
        }
        $this->printMessage("EXAMPLES - Movements for Draft Orders:\n");
        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("SKU: {$result->product->sku}, Order {$result->link->salesOrder->sales_order_number}");
        }
    }

    private function zeroUnitsMovementsExamples()
    {
        $query = InventoryMovement::with(['product'])->where('quantity', 0);
        if ($query->count() > 0) {
            return;
        }

        $this->printMessage("EXAMPLES - Zero Unit Movements:\n");

        foreach ($query->take(5)->get() as $result) {
            $this->printMessage("Movement ID: $result->id, SKU: {$result->product->sku}, Reference: $result->reference, Link Type: $result->link_type");
        }
    }

    private function unmatchedMovementsAndLineQuantitiesExamples()
    {
        $query = $this->unmatchedMovementsAndLineQuantitiesQuery();
        if ($query->count() == 0) {
            return;
        }
        $this->printMessage("EXAMPLES - Unmatched Movement & Line Quantities:\n");
        foreach ($query->take(5)->get() as $result) {
            /** @var SalesOrderLine $salesOrderLine */
            $salesOrderLine = SalesOrderLine::with([])->findOrFail($result->id);
            $this->printMessage("{$salesOrderLine->salesOrder->sales_order_number} (SKU: {$salesOrderLine->product->sku}), Line qty: {$result->sol_qty}, Reserved qty: $result->reserved_qty, Active qty: $result->active_qty ");
        }
    }

    public function remedy(): void
    {
        $this->console->info('Remedies for mismatched sales order line and movement quantities');
        $this->console->info("\tFix movements for draft orders");
        $this->fixMovementsForDraftOrders();
        $this->console->info("\tFix unmatched movement and line quantities");
        $this->fixUnmatchedMovementAndLineQuantities();
        $this->console->info("\tFix movements with zero quantity");
        $this->fixMovementsWithZeroQuantity();
        $this->console->info("\tFix lines without zero sum movements");
        $this->fixLinesWithoutZeroSumMovements();
        $this->console->info("\tFix fulfillment lines without movements");
        $this->fixFulfillmentLinesWithoutMovements();
    }

    /**
     * @throws OversubscribedFifoLayerException
     */
    public function fixFulfillmentLinesWithoutMovements(): void
    {
        $this->fulfillmentLinesWithoutMovementsQuery()->each(function (SalesOrderFulfillmentLine $fulfillmentLine) {
            $this->console->info("ID: $fulfillmentLine->id Qty: $fulfillmentLine->quantity");
            if ($fulfillmentLine->fulfilled_qty - $fulfillmentLine->movement_qty < 0) {
                $this->console->info("Fulfilled quantity is higher than movement quantity for sofl: $fulfillmentLine->id");

                return;
            }

            if ($fulfillmentLine->fulfilled_qty == $fulfillmentLine->movement_qty) {
                $this->console->info("Fulfilled quantity is equal to the movement quantity for sofl: $fulfillmentLine->id");

                return;
            }

            $this->console->info($fulfillmentLine->salesOrderLine->salesOrder->sales_order_number.' - '.$fulfillmentLine->salesOrderLine->product->sku);
            if ($this->debugging() && ! $this->console->confirm('Would you like to reverse the reservation for any fulfilled quantity?')) {
                return;
            }

            DB::beginTransaction();
            try {
                $this->console->info("Sales Order Line ID: $fulfillmentLine->sales_order_line_id");
                // Consider possibility that reservation movements exist, but they are backorder ones.  In this case the fulfillments should be reversed.
                $quantityReservedFifo = $fulfillmentLine->salesOrderLine->inventoryMovements->where('layer_type', FifoLayer::class)
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->sum('quantity');
                $quantityReservedBackorder = $fulfillmentLine->salesOrderLine->inventoryMovements->where('layer_type', BackorderQueue::class)
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->sum('quantity');
                $this->console->info("Quantity Reserved FIFO: $quantityReservedFifo, Quantity Reserved Backorder: $quantityReservedBackorder");
                $netQuantityReserved = $quantityReservedFifo - $quantityReservedBackorder;
                // Insufficient sales order line reservations
                if ($fulfillmentLine->quantity > $netQuantityReserved) {
                    if ($this->debugging() && ! $this->console->confirm('Would you like to delete the fulfillment if quantity fulfilled is higher than the net reserved quantity for the sales order line?')) {
                        $fulfillmentLine->negateReservationMovements(quantity: $fulfillmentLine->fulfilled_qty - $fulfillmentLine->movement_qty);
                        DB::commit();

                        return;
                    }
                    $this->console->info("Fulfillment quantity is higher than net quantity reserved for sales order line with fulfillment line id: $fulfillmentLine->id, so deleting fulfillment");
                    $this->console->warn("Disabled deletion of fulfillment... this logic needs review");
                    return;
                    // TODO: Do or summarize positive adjustments needed instead.

                    $fulfillmentLine->salesOrderFulfillment->delete(true);
                    $salesOrder = $fulfillmentLine->salesOrderLine->salesOrder;
                    $salesOrder->refresh();
                    $fulfillmentLine->salesOrderLine->salesOrder->update(['order_status' => SalesOrder::STATUS_OPEN, 'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED]);
                    DB::commit();
                } else {
                    $reservationFifoLayer = $fulfillmentLine->salesOrderLine
                        ->inventoryMovements()
                        ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                        ->first()
                        ->layer;
                    InventoryMovement::create([
                        'inventory_movement_date' => $fulfillmentLine->salesOrderFulfillment->fulfilled_at,
                        'product_id' => $fulfillmentLine->salesOrderLine->product_id,
                        'warehouse_id' => $fulfillmentLine->salesOrderLine->warehouse_id,
                        'link_id' => $fulfillmentLine->id,
                        'link_type' => SalesOrderFulfillmentLine::class,
                        'layer_id' => $reservationFifoLayer->id,
                        'layer_type' => FifoLayer::class,
                        'quantity' => -$fulfillmentLine->quantity,
                        'inventory_status' => InventoryMovement::INVENTORY_STATUS_RESERVED,
                        'type' => InventoryMovement::TYPE_SALE,
                    ]);
                    app(FifoLayerRepository::class)->validateFifoLayerCache($reservationFifoLayer);
                }
            } catch (Throwable $e) {
                DB::rollBack();
                $this->console->info($e->getMessage());
            }
        });
    }

    private function fixMovementsForDraftOrders(): void
    {
        $this->deleteMovements($this->movementsForDraftOrdersQuery()->get());
    }

    private function fixLinesWithoutZeroSumMovements(): void
    {
        /** @var SalesOrderLine $salesOrderLine */
        foreach ($this->nonZeroSumReservationsQuery(['inventoryMovements'])->get() as $salesOrderLine) {
            if ($salesOrderLine->no_audit_trail) {
                // We delete the movements as they shouldn't exist
                // in the first place.
                $this->deleteMovements($salesOrderLine->inventoryMovements);

                return;
            }

            // We fix missing reservations.
            try {
                DB::beginTransaction();

                // Fix missing reservation movements.
                $activeMovements = $salesOrderLine->inventoryMovements()
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->get();

                /** @var InventoryMovement $activeMovement */
                foreach ($activeMovements as $activeMovement) {
                    $reservation = $salesOrderLine->inventoryMovements()
                        ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                        ->where('layer_type', $activeMovement->layer_type)
                        ->where('layer_id', $activeMovement->layer_id)
                        ->where('quantity', abs($activeMovement->quantity))
                        ->first();
                    if (! $reservation) {
                        // We create the reservation.
                        $reservation = $activeMovement->replicate();
                        $reservation->quantity = abs($activeMovement->quantity);
                        $reservation->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                        $salesOrderLine->inventoryMovements()->save($reservation);
                    }
                }

                // Fix missing active movements.
                $reservationMovements = $salesOrderLine->inventoryMovements()
                    ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->get();

                /** @var InventoryMovement $reservationMovement */
                foreach ($reservationMovements as $reservationMovement) {
                    $active = $salesOrderLine->inventoryMovements()
                        ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                        ->where('layer_type', $reservationMovement->layer_type)
                        ->where('layer_id', $reservationMovement->layer_id)
                        ->where('quantity', -abs($reservationMovement->quantity))
                        ->first();
                    if (! $active) {
                        $this->console->info("Missing active inventory movement for SOL: $salesOrderLine->id");
                        // Reservation fifo layer does not have available quantity, so get the available quantity to use
                        if ($reservationMovement->layer_type !== BackorderQueue::class && ! $this->isFifoDeficientBy($reservationMovement->fifo_layer, $reservationMovement->quantity)) {
                            $this->console->info("Reservation Layer Type: $reservationMovement->layer_type");
                            $this->console->info("Reservation movement FIFO layer does not have enough inventory for the available quantity so moving to available layer(s)).  Need: $reservationMovement->quantity, Available: {$reservationMovement->fifo_layer->available_quantity}");
                            $getFifoLayersForQuantity = app(GetFifoLayersForQuantity::class);
                            $fifoLayersForQuantity = $getFifoLayersForQuantity(
                                $reservationMovement->fifo_layer->product,
                                $reservationMovement->fifo_layer->warehouse,
                                $reservationMovement->quantity
                            )->fifoLayerQuantities;

                            $fulfillmentMovements = collect();
                            $fifoLayersForQuantity->each(function (FifoLayerQuantityData $fifoLayerData) use ($salesOrderLine, $reservationMovement, $fifoLayersForQuantity, &$fulfillmentMovements) {

                                $activeInventoryMovement = $reservationMovement->replicate();
                                $activeInventoryMovement->quantity = -$fifoLayerData->quantity;
                                $activeInventoryMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                                $activeInventoryMovement->layer_id = $fifoLayerData->fifoLayer->id;
                                $activeInventoryMovement->save();

                                // This condition must pass on the first iteration of the "each" iteration
                                if ($fifoLayersForQuantity->first()->fifoLayer->id === $fifoLayerData->fifoLayer->id) {
                                    $fulfillmentMovements = app(InventoryMovementRepository::class)->getCorrespondingFulfillmentMovementsForReservationSalesOrderLineMovement($reservationMovement);

                                    $fulfillmentMovements->each(function (InventoryMovement $fulfillmentMovement) use ($fifoLayerData) {
                                        $fulfillmentMovement->quantity = -$fifoLayerData->quantity;
                                        $fulfillmentMovement->layer_id = $fifoLayerData->fifoLayer->id;
                                        $fulfillmentMovement->save();
                                    });

                                    $reservationMovement->quantity = $fifoLayerData->quantity;
                                    $reservationMovement->layer_id = $fifoLayerData->fifoLayer->id;
                                    $reservationMovement->save();

                                    return;
                                }

                                // At this point we know there was more than one fifo layer needed for the active movement and the
                                // existing reservation and fulfillment movement already got moved to the first fifo layer.  Now
                                // we have to create the new movements for each

                                $newReservationMovement = $activeInventoryMovement->replicate();
                                $newReservationMovement->quantity = $fifoLayerData->quantity;
                                $newReservationMovement->inventory_status = InventoryMovement::INVENTORY_STATUS_RESERVED;
                                $newReservationMovement->layer_id = $fifoLayerData->fifoLayer->id;
                                $newReservationMovement->save();

                                $fulfillmentMovements->each(function (InventoryMovement $fulfillmentMovement) use ($fifoLayerData, $newReservationMovement) {
                                    $newFulfillmentMovement = $newReservationMovement->replicate();
                                    $newFulfillmentMovement->quantity = -$fifoLayerData->quantity;
                                    $newFulfillmentMovement->layer_id = $fifoLayerData->fifoLayer->id;
                                    $newFulfillmentMovement->link_type = $fulfillmentMovement->link_type;
                                    $newFulfillmentMovement->link_id = $fulfillmentMovement->link_id;
                                    $newFulfillmentMovement->save();
                                });

                                SalesOrderLineLayer::create([
                                    'sales_order_line_id' => $salesOrderLine->id,
                                    'quantity' => $fifoLayerData->quantity,
                                    'layer_id' => $fifoLayerData->fifoLayer->id,
                                    'layer_type' => FifoLayer::class,
                                ]);

                                app(FifoLayerRepository::class)->validateFifoLayerCache($fifoLayerData->fifoLayer);
                            });

                            continue;
                        }

                        $this->console->info("Reservation movement FIFO layer has enough inventory so creating the available movement on the reservation fifo layer");

                        $active = $reservationMovement->replicate();
                        $active->quantity = -abs($reservationMovement->quantity);
                        $active->inventory_status = InventoryMovement::INVENTORY_STATUS_ACTIVE;
                        $salesOrderLine->inventoryMovements()->save($active);

                        SalesOrderLineLayer::create([
                            'sales_order_line_id' => $salesOrderLine->id,
                            'quantity' => abs($reservationMovement->quantity),
                            'layer_id' => $reservationMovement->fifoLayer->id,
                            'layer_type' => FifoLayer::class,
                        ]);

                        app(FifoLayerRepository::class)->validateFifoLayerCache($reservationMovement->fifoLayer);
                    }
                }

                $totalActive = abs($salesOrderLine->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
                    ->sum('quantity'));

                $totalReserved = $salesOrderLine->inventoryMovements()->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
                    ->where('quantity', '>', 0)
                    ->sum('quantity');
                if ($totalReserved != $totalActive) {
                    throw new Exception("Non-zero sum movements for SOL: $salesOrderLine->id");
                }

                if ($totalActive != $salesOrderLine->quantity) {
                    //throw new Exception("Unequal movement qty and line quantity for SOL: $salesOrderLine->id belonging to {$salesOrderLine->salesOrder->sales_order_number} for {$salesOrderLine->product->sku}");
                    $this->console->info("Unequal movement qty and line quantity for SOL: $salesOrderLine->id belonging to {$salesOrderLine->salesOrder->sales_order_number} for {$salesOrderLine->product->sku}, skipping");

                    continue;
                }

                DB::commit();
            } catch (Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine());
            }
        }
    }

    public function isFifoDeficientBy(FifoLayer $fifoLayer, int $quantity): bool
    {
        // The fifo layer should have missing active reservation to the
        // tune of the provided quantity.
        $usageByMovements = abs($fifoLayer->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('quantity', '<', 0)
            ->sum('quantity'));

        $this->console->info("Layer ID: $fifoLayer->id, Reservation Fifo Layer Available Quantity: $fifoLayer->available_quantity, Sum of Available Quantity: $usageByMovements, Quantity Needed: $quantity");
        return $fifoLayer->original_quantity - $usageByMovements >= $quantity;
    }

    private function deleteMovements(\Illuminate\Support\Collection $movements): void
    {
        /** @var InventoryMovement $movement */
        foreach ($movements as $movement) {
            DB::transaction(function () use ($movement) {
                $movement->delete();
                $movement->reverseLayer();
            });
        }
    }

    private function fixMovementsWithZeroQuantity(): void
    {
        InventoryMovement::query()->where('quantity', 0)->delete();
    }

    private function fixUnmatchedMovementAndLineQuantities(): void
    {
        $results = $this->unmatchedMovementsAndLineQuantitiesQuery()->get();

        foreach ($results as $result) {
            DB::beginTransaction();
            try {
                /** @var SalesOrderLine $salesOrderLine */
                $salesOrderLine = SalesOrderLine::find($result->id);
                $product = $salesOrderLine->product;
                $salesOrder = $salesOrderLine->salesOrder;

                $this->console->info('Sales Order '.$salesOrder->sales_order_number.', SKU: '.$product->sku.' (Qty: '.$salesOrderLine->quantity.') (ID: '.$salesOrderLine->id.')');

                /*
                 * Delete existing reservation movements
                 */

                $movements = $salesOrderLine->inventoryMovements->where('type', 'sale')->where('link_type', SalesOrderLine::class)->groupBy('layer_id');

                foreach ($movements as $layer_movements) {
                    foreach ($layer_movements as $movement) {
                        $this->console->info("\t".$movement->quantity." ($movement->inventory_status) belonging to layer $movement->layer_id ($movement->layer_type)");
                    }
                }

                if ($this->debugging() && ! $this->console->confirm('Would you like to delete these movements?')) {
                    echo 'OK, skipping'."\n";

                    continue;
                }

                foreach ($movements as $layer_movements) {
                    /** @var InventoryMovement $movement */
                    foreach ($layer_movements as $movement) {
                        $movement->delete();
                        $movement->reverseLayer();
                    }
                }
                (new UpdateProductsInventoryAndAvgCost([$salesOrderLine->product_id]))->handle();

                /*
                 * Recreate reservation movements
                 */

                if ($this->debugging() && ! $this->console->confirm('Recreate reservations?')) {
                    continue;
                }

                $manager = new InventoryManager($salesOrderLine->warehouse_id, $salesOrderLine->product);

                try {
                    $manager->takeFromStock($salesOrderLine->quantity, $salesOrderLine);
                } catch (InsufficientStockException $ie) {
                    $this->handleLineAdjustments($salesOrderLine, $salesOrderLine->quantity);
                    $manager->takeFromStock($salesOrderLine->quantity, $salesOrderLine);
                }

                DB::commit();
            } catch (Throwable $e) {
                DB::rollBack();
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTraceAsString());
            }
        }
    }
}
