<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Console\Commands\Inventory\Integrity\Contracts\Remedy;
use App\Exceptions\InsufficientMovableNegativeInventoryMovementsException;
use App\Exceptions\InsufficientStockException;
use App\Jobs\UpdateProductsInventoryAndAvgCost;
use App\Models\BackorderQueue;
use App\Models\BackorderQueueRelease;
use App\Models\FifoLayer;
use App\Models\InventoryAdjustment;
use App\Models\InventoryAssemblyLine;
use App\Models\InventoryMovement;
use App\Models\PurchaseOrderShipmentReceiptLine;
use App\Models\SalesCreditReturnLine;
use App\Models\StockTakeItem;
use App\Models\WarehouseTransferShipmentReceiptLine;
use App\Services\InventoryManagement\Actions\FixInvalidFifoLayerFulfilledQuantityCache;
use Exception;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;

class InvalidFifoLayerFulfilledQuantityCache extends Integrity implements Identifier, Remedy
{
    private FixInvalidFifoLayerFulfilledQuantityCache $fixer;

    public function __construct(
        protected Command $console
    ) {
        parent::__construct($console);
        $this->fixer = app(FixInvalidFifoLayerFulfilledQuantityCache::class);
    }

    public function invalidFifoLayerFulfilledQuantityCache(): Builder
    {
        return InventoryMovement::query()
            ->selectRaw('fifo_layers.id, fifo_layers.available_quantity, sum(inventory_movements.quantity) as usage_quantity, (fifo_layers.available_quantity - sum(inventory_movements.quantity)) as shortage_quantity, products.id as product_id, products.sku')
            ->join('fifo_layers', 'fifo_layers.id', '=', 'inventory_movements.layer_id')
            ->join('products', 'products.id', '=', 'fifo_layers.product_id')
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)
            ->where('layer_type', FifoLayer::class)
            ->groupBy('layer_id')
            ->havingRaw('SUM(inventory_movements.quantity) <> fifo_layers.available_quantity');
    }

    public function overReleasedBackorders(): Builder
    {
        return BackorderQueue::query()
            ->selectRaw(
                "backorder_queues.id, abs(inventory_movements.quantity) - backorder_queues.shortage_quantity as over_released_quantity, products.id as product_id, products.sku"
            )
            ->join(
                "inventory_movements",
                "inventory_movements.layer_id",
                "=",
                "backorder_queues.id"
            )
            ->join("products", "products.id", "=", "inventory_movements.product_id")
            ->where("inventory_movements.layer_type", BackorderQueue::class)
            ->where(
                "inventory_movements.inventory_status",
                InventoryMovement::INVENTORY_STATUS_ACTIVE
            )
            ->whereRaw(
                "abs(inventory_movements.quantity) - backorder_queues.shortage_quantity <> 0"
            );
    }

    private function positiveInventoryEventsWithoutMovements(): void
    {
        $this->checkForLinesWithoutMovements(WarehouseTransferShipmentReceiptLine::class);
        $this->checkForLinesWithoutMovements(SalesCreditReturnLine::class);
        $this->checkForLinesWithoutMovements(PurchaseOrderShipmentReceiptLine::class);
        $this->checkForLinesWithoutMovements(InventoryAdjustment::class);
        $this->checkForLinesWithoutMovements(StockTakeItem::class);
        $this->checkForLinesWithoutMovements(InventoryAssemblyLine::class);
    }

    private function checkForLinesWithoutMovements(string $modelClass): void
    {
        $linesWithoutMovements = app($modelClass)
            ->whereDoesntHave('inventoryMovements');

        if ($modelClass == WarehouseTransferShipmentReceiptLine::class) {
            // TODO: SKU-6213 this needs to be handled/fixed.  The following is just a band aid until fixed.
            $linesWithoutMovements->where('quantity', '>', 0);
        }

        if ($modelClass == SalesCreditReturnLine::class) {
            $linesWithoutMovements->whereIn('action', SalesCreditReturnLine::ACTION_INVENTORY_IMPACT);
        }
        if ($modelClass == InventoryAdjustment::class) {
            $linesWithoutMovements->where('quantity', '>', 0);
        }
        if ($modelClass == StockTakeItem::class) {
            $linesWithoutMovements->whereColumn('snapshot_inventory', '!=', 'qty_counted');
        }

        $linesWithoutMovementsCount = $linesWithoutMovements->count();

        if ($linesWithoutMovementsCount > 0) {
            $this->console->warn("There are $linesWithoutMovementsCount $modelClass without inventory movements.");
        }
    }

    public function identify(): void
    {
        $this->console->info('Checking for positive inventory events without movements...');
        $this->positiveInventoryEventsWithoutMovements();

        $total = $this->invalidFifoLayerFulfilledQuantityCache()->count();
        $this->addMessage('There are '.$total.' FIFO with invalid cache', $total);
    }

    public function examples(): void
    {
        $query = $this->invalidFifoLayerFulfilledQuantityCache();
        if ($query->count() == 0) {
            return;
        }
        $this->printMessage("EXAMPLES: Invalid fifo layer fulfilled quantity cache\n");
        foreach ($query->take(10)->get() as $result) {
            $this->printMessage("SKU: {$result->sku}, Fifo Layer ID: $result->id");
        }
    }

    /**
     * @throws Exception
     */
    public function remedy(): void
    {
        /*
         * First must update/remove backorder releases and update backorder queue released_quantity cache for backorders
         * with inventory movements because releases should have gotten rid of the backordered inventory movement and if the
         * backordered inventory movement still exists, it wasn't released.
         */
        $overReleasedBackorders = $this->overReleasedBackorders()->get();
        /** @var BackorderQueue $backorder */
        foreach ($overReleasedBackorders as $backorder)
        {
            $backorderQueue = BackorderQueue::find($backorder->id);
            $overReleasedQuantity = $backorder->over_released_quantity;
            $totalReleasedQuantity = $backorder->backorderQueueReleases->sum('released_quantity');

            if ($overReleasedQuantity == $totalReleasedQuantity)
            {
                $this->console->info("Removing backorder releases for backorder: $backorder->id");
                $backorderQueue->backorderQueueReleases()->delete();
                $backorderQueue->released_quantity = 0;
                $backorderQueue->save();
            } elseif ($overReleasedQuantity <= $totalReleasedQuantity)
            {
                throw new Exception("Backorder: $backorder->id has more releases than over released quantity.  Adjustment of releases is needed.");
            } else {
                $this->console->info("Updating released quantity cache for backorder: $backorder->id");
                $backorderQueue->released_quantity = 0;
                $backorderQueue->save();
            }
        }

        $shortages = [];
        $invalidFifoLayers = $this->invalidFifoLayerFulfilledQuantityCache()->get();
        foreach ($invalidFifoLayers as $fifoLayer) {
            try {
                $this->console->warn("Fixing fifo layer: $fifoLayer->id with shortage of $fifoLayer->shortage_quantity");
                /** @var FifoLayer $fifoLayer */
                $fifoLayer = FifoLayer::query()->findOrFail($fifoLayer->id);

                $this->fixer->fix($fifoLayer);
                $this->console->info("Fifo layer: $fifoLayer->id fixed.");
            } catch (InsufficientStockException $e) {
                $shortages[] = $fifoLayer->sku.','.$fifoLayer->id;
            } catch (InsufficientMovableNegativeInventoryMovementsException $e) {
                $this->console->warn("Fifo layer: $fifoLayer->id has insufficient movable negative inventory movements.");
            } catch (\Throwable $e) {
                dd($e->getMessage(), $e->getFile(), $e->getLine(), $e->getTrace());
            }
        }
        if (count($shortages) > 0) {
            $this->console->warn('Shortages:');
            foreach ($shortages as $shortage) {
                $this->console->warn($shortage);
            }
        }
        dispatch(new UpdateProductsInventoryAndAvgCost($invalidFifoLayers->pluck('product_id')->toArray()))->onQueue('syncInventory');
    }

    public function description(): string
    {
        return 'Fix invalid fifo layers fulfilled quantity cache';
    }
}
