<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Models\InventoryAdjustment;
use App\Models\SalesOrderLine;
use App\Services\InventoryManagement\Actions\FixInvalidFifoLayerFulfilledQuantityCache;
use App\Services\InventoryManagement\InventoryManager;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Throwable;

class Integrity
{
    private array $messages;

    public function __construct(protected Command $console)
    {
    }

    protected function addMessage(string $message, ?int $count = 0): void
    {
        $this->messages[] = [
            'message' => $message,
            'count' => $count,
        ];
    }

    protected function printMessage(string $message): void
    {
        $this->console->info($message);
    }

    protected function debugging(): bool
    {
        return $this->console->option('debug');
    }

    public function messages(): array
    {
        return $this->messages;
    }

    /**
     * Line adjustments are for when there is insufficient stock to fulfill a sales order line.
     * and we want to handle this by deleting a previous negative adjustment or creating a positing adjustment
     *
     * @throws Throwable
     */
    protected function handleLineAdjustments(SalesOrderLine $salesOrderLine, int $applicableQty)
    {
        $this->console->info('Handling line adjustments for '.$salesOrderLine->id);
        $product = $salesOrderLine->product;

        /*
         * Applicable quantity should subtract any unfulfilled quantity that is available from FIFO layers.
         * Note this fails when the unfulfilled quantity is outdated.
         */
        $originalApplicableQty = $applicableQty;
        $applicableQty -= $product->activeFifoLayers->sum('available_quantity');

        if ($applicableQty <= 0) {
            $this->console->info('No applicable quantity.  Going to run FIFO Layer Fix Cache');
            $product->activeFifoLayers->filter(fn ($fifoLayer) => $fifoLayer->available_quantity > 0)->each(function ($fifoLayer) {
                $this->console->info('Fixing FIFO Layer Cache for '.$fifoLayer->id);
                app(FixInvalidFifoLayerFulfilledQuantityCache::class)->fix($fifoLayer);
            });
            $product->refresh();
            $applicableQty = $originalApplicableQty - $product->activeFifoLayers->sum('available_quantity');
            if ($applicableQty <= 0) {
                $this->console->info('No applicable quantity after FIFO Layer Fix Cache');

                return;
            }
        }

        $adjustmentNumber = 0;

        do {
            // Take stock from most recent negative adjustment
            // for the same warehouse.
            /** @var InventoryAdjustment $adjustment */
            $adjustment = $product->inventoryAdjustments()
                ->where('quantity', '<', 0)
                ->where('warehouse_id', $salesOrderLine->warehouse_id)
                ->orderBy('adjustment_date', 'DESC')
                ->offset($adjustmentNumber)
                ->first();

            if (! $adjustment) {
                $this->console->info('No more adjustments to take stock from.');
                break;
            }

            if (abs($adjustment->quantity) <= $applicableQty) {
                // We delete the adjustment and
                // update the applicable qty.
                $message = "Delete Adjustment -> ID: $adjustment->id, Qty: $adjustment->quantity, Notes: $adjustment->notes?";
                if ($this->debugging() && ! $this->console->confirm($message)) {
                    $adjustmentNumber++;

                    continue;
                }
                $adjustment->delete(false);
                $applicableQty -= abs($adjustment->quantity);
            } else {
                break;
            }
        } while ($applicableQty > 0);

        // If more quantity is needed, we create a positive adjustment
        if ($applicableQty > 0) {
            $message = "Create Adjustment -> Qty: $applicableQty for $product->sku?";
            if ($this->debugging() && ! $this->console->confirm($message)) {
                return;
            }

            $adjustment = new InventoryAdjustment();
            $adjustment->quantity = $applicableQty;
            $adjustment->adjustment_date = Carbon::now();
            $adjustment->notes = 'sku.io integrity.';
            $adjustment->warehouse_id = $salesOrderLine->warehouse_id;
            $product->inventoryAdjustments()->save($adjustment);

            InventoryManager::with(
                $adjustment->warehouse_id,
                $adjustment->product
            )->addToStock(abs($adjustment->quantity), $adjustment, true, false);
        }
    }
}
