<?php

namespace App\Console\Commands\Inventory\Integrity;

use App\Console\Commands\Inventory\Integrity\Contracts\Identifier;
use App\Notifications\MonitoringMessage;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Notification;

class Identify extends Command
{
    const SLACK_ALERTS_CHANNEL = 'slack.alerts';

    protected $signature = 'sku:inventory:integrity:identify
                                {--debug : Debug mode }
                                {--i : Interactive mode }';

    protected $description = 'Identifies inventory integrity issues.';

    protected array $identifiers;

    public function __construct()
    {
        parent::__construct();

        $this->identifiers = [
            new InvalidFulfilledQuantityCache($this),
            // Disable until refactor of Shopify integration
            //new UnprocessedSalesCredits($this),
            /*
             * This identifier is disabled because we can't reliably detect the issue.  Also a future remedy may be to just mark
             * as out of sync.  Note example BG110499 which has 2 separate fulfillments and is producing a false positive.  Also
             * note that Non product lines being fulfilled may also produce a false positive for being out of sync.
             */
            //new SalesChannelAndSkuFulfillmentsMismatch($this),
            new MismatchedSalesOrderLineAndMovementQuantities($this),
            new MissingReservations($this),
            new NoAuditTrailLinesWithMovements($this),
            new OrphanedReservations($this),
            new InvalidFifoLayerFulfilledQuantityCache($this),
            new PreAuditTrailReservations($this),
            new OutOfSyncInventoryMovementReferences($this),
        ];
    }

    public function handle(): int
    {
        $interactive = $this->option('i');

        if ($interactive) {
            $headers = ['No.', 'Identifier'];

            $data = array_map(function ($id, Identifier $identifier) {
                return [$id, get_class($identifier)];
            }, array_keys($this->identifiers), $this->identifiers);

            $this->table($headers, $data);

            $identifierIndexInput = $this->ask('Enter numbers of identifiers to run (comma separated) or "all" to process all:');
        }

        if (isset($identifierIndexInput) && $identifierIndexInput !== 'all') {
            $identifiersToRun = array_map('intval', explode(',', str_replace(' ', '', $identifierIndexInput)));
        } else {
            $identifiersToRun = array_keys($this->identifiers);
        }

        foreach ($this->identifiers as $index => $identifier) {
            if (! in_array($index, $identifiersToRun)) {
                continue;
            }

            $identifier->identify();
            $this->report($identifier->messages());

            if (collect($identifier->messages())->sum('count') > 0 && $this->debugging()) {
                if ($this->confirm('Do you want to see some examples?')) {
                    $identifier->examples();
                }
            }
        }

        return 0;
    }

    protected function debugging(): bool
    {
        return $this->option('debug');
    }

    protected function report(array $results): void
    {
        foreach ($results as $result) {
            $this->alertMessage($result['message'], $result['count'] ?? 0);
        }
    }

    protected function alertMessage(string $message, int $count = 0): void
    {
        $this->info($message);
        if ($count > 0 && ! $this->option('debug')) {
            // Don't send alerts for now until we have time to handle this
            //$this->alertSku($message);
        }
    }

    protected function alertSku(string $message): void
    {
        $message = "SKU INVENTORY INTEGRITY:\n$message\n".env('APP_URL', 'SKU.io');
        Notification::route('slack', config(self::SLACK_ALERTS_CHANNEL))
            ->notify(new MonitoringMessage($message));
    }
}
