<?php

namespace App\Console\Commands\Inventory\Health\Fix\Traits;

use App\Models\SalesOrderLine;
use Illuminate\Support\Collection;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\search;
use function Laravel\Prompts\table;

trait FixSalesOrderLinesTrait
{
    public function showProblemRecords(Collection $data): void
    {
        if (confirm("Show problem records?")) {
            $headers = [
                'Sales Order Line ID',
                'Sales Order ID',
                'Order #',
                'Qty',
                'Product ID',
                'SKU',
                'Warehouse',
            ];

            $rows = $data->map(function(SalesOrderLine $row) {
                return [
                    $row->id,
                    $row->sales_order_id,
                    $row->salesOrder->sales_order_number,
                    $row->quantity,
                    $row->product->id,
                    $row->product->sku,
                    $row->warehouse->name,
                ];
            })->toArray();

            table(
                headers: $headers,
                rows: $rows
            );

            $this->sendCsv($headers, $rows);
        }
    }

    protected function filter(): void
    {
        if (confirm(
            label: "Would you like to fix for all orders or a specific one?",
            yes: "All",
            no: "Specific Sales Order",
        )) {
            info("Fixing for all sales orders");
            // Fix for all
        } else {
            $salesOrderId = search(
                label: "Search for a sales order number",
                options: fn(string $value) => $this->data->filter(
                    fn(SalesOrderLine $salesOrderLine) => stripos($salesOrderLine->salesOrder->sales_order_number, $value) !== false
                )->pluck('salesOrder.sales_order_number', 'sales_order_id')->all(),
            );
            $this->data = $this->data->where('sales_order_id', $salesOrderId);
            $salesOrder = $this->data->first();
            info("Fixing for sales order $salesOrder->sales_order_number");
        }
    }
}
