<?php

namespace App\Console\Commands\Inventory\Health\Fix;

use App\Console\Commands\Inventory\Health\AbstractInventoryHealthCommandHelper;
use App\Console\Commands\Inventory\Health\Fix\Traits\FixInventoryMovementsTrait;
use App\Models\Product;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Throwable;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\search;
use function Laravel\Prompts\select;
use function Laravel\Prompts\table;

class FixNonInventoryProductsWithInventoryMovementsCommandHelper extends AbstractInventoryHealthCommandHelper
{
    use FixInventoryMovementsTrait;

    public function getDescription(): string
    {
        return 'Products that are non-inventory (Bundle/Matrix) that have inventory movements';
    }

    public function getQuery(): Builder
    {
        return $this->health->getNonInventoryProductsWithInventoryMovementsQuery();
    }

    /** @var Product $dataItem */
    protected function getFixLabel(mixed $dataItem): string
    {
        return "Fixing for Product ID {$dataItem->id}, SKU {$dataItem->sku}";
    }

    /**
     * @throws Throwable
     * @var Product $dataItem
     */
    protected function fix($dataItem): void
    {
        $this->healthManager->fixNonInventoryProductWithInventoryMovements($dataItem);
    }

    public function showProblemRecords(Collection $data): void
    {
        if (confirm("Show problem records?")) {
            $headers = [
                'ID',
                'SKU',
                'Name',
                'FIFO Count',
                'Backorder Count',
                'Movement Count',
            ];

            $rows = $data->map(function(Product $row) {
                return [
                    $row->id,
                    $row->sku,
                    $row->name,
                    $row->fifo_layers_count,
                    $row->backorder_queues_count,
                    $row->inventory_movements_count,
                ];
            })->toArray();

            table(
                headers: $headers,
                rows: $rows
            );

            $this->sendCsv($headers, $rows);
        }
    }

    protected function filter(): void
    {
        $fixSelection = select(
            label: "Which Products would you like to fix?",
            options: [
                'all' => 'All',
                'product' => 'Specific product',
            ],
        );

        if ($fixSelection === 'product') {
            $productId = search(
                label: "Search for a product SKU",
                options: fn(string $value) => $this->data->filter(
                    fn($product) => stripos($product->sku, $value) !== false
                )->pluck('sku', 'id')->all(),
            );
            $this->data = $this->data->where('id', $productId);
        }
    }
}