<?php

namespace App\Console\Commands\Inventory\Health\Fix;

use App\Console\Commands\Inventory\Health\Fix\Traits\FixSalesOrderLinesTrait;
use App\Console\Commands\Inventory\Health\AbstractInventoryHealthCommandHelper;
use App\Data\ProductWarehouseQuantityData;
use App\Exceptions\CantBackorderAlreadyFulfilledQuantityException;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Models\StockTake;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Throwable;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\text;

class FixMissingSalesOrderMovementsCommandHelper extends AbstractInventoryHealthCommandHelper
{
    use FixSalesOrderLinesTrait;

    protected Collection $productWarehouseQuantities;

    public function __construct(Command $console)
    {
        parent::__construct($console);
        $this->productWarehouseQuantities = collect();
    }

    public function getDescription(): string {
        return 'Sales order lines without active inventory movements (Closed only)';
    }

    public function getQuery(): Builder
    {
        return $this->health->salesOrderLinesMissingMovementsQuery(InventoryMovement::INVENTORY_STATUS_ACTIVE);
    }

    /** @var SalesOrderLine $dataItem */
    protected function getFixLabel(mixed $dataItem): string
    {
        return "Fixing for sales order {$dataItem->salesOrder->sales_order_number} line $dataItem->id with SKU {$dataItem->product->sku} for quantity $dataItem->quantity";
    }

    /**
     * @throws Throwable
     * @var SalesOrderLine $dataItem
     */
    protected function fix($dataItem): void
    {
        $reservationMovements = $dataItem->inventoryMovements()
            ->where('inventory_status', InventoryMovement::INVENTORY_STATUS_RESERVED)
            ->get();
        if ($reservationMovements->isEmpty()) {
            customlog("inventory-fixes", "Sales order line $dataItem->id is also missing reserved movements... including those in the fix", days: null);
            try {
                $this->healthManager->createSalesOrderReservation($dataItem);
            } catch (CantBackorderAlreadyFulfilledQuantityException $e) {
                customlog("inventory-fixes", $e->getMessage(), days: null);
                $this->productWarehouseQuantities->add(ProductWarehouseQuantityData::from([
                    'product_id' => $dataItem->product_id,
                    'warehouse_id' => $dataItem->warehouse_id,
                    'quantity' => $dataItem->quantity
                ]));
                return;
            }
        } else {
            $reservationMovements->each(fn(InventoryMovement $reservationMovement) =>
            $this->healthManager->createActiveInventoryMovementFromReserved($reservationMovement)
            );
        }
        if ($this->health->salesOrderLinesMissingFulfillmentMovementsQuery($dataItem)->exists()) {
            customlog("inventory-fixes", "Sales order line $dataItem->id is also missing fulfillment movements... including those in the fix", days: null);
            $this->healthManager->createFulfillmentInventoryMovementsForSalesOrderLine($dataItem);
        }
    }

    protected function postFix(): bool
    {
        if ($this->productWarehouseQuantities->isNotEmpty())
        {
            if (confirm('Would you like to create a stock take for the missing fulfillment inventory movements so that the missing movements fix can be ran again?')) {
                $notes = text('Please enter notes for the stock take');
                $stockTakesCreated = $this->healthManager->createStockTakeForProductWarehouseQuantities($this->productWarehouseQuantities, $notes);
                if ($stockTakesCreated->isEmpty()) {
                    info('No stock takes were created');
                    return false;
                } else {
                    $stockTakesCreated->each(function (StockTake $stockTake) {
                        info("Stock take created (ID: $stockTake->id) for warehouse {$stockTake->warehouse->name}");
                    });
                    info("Retrying missing movement fixes...");
                    $this->productWarehouseQuantities = collect();
                    return true;
                }
            }
        }
        return false;
    }
}