<?php

namespace App\Console\Commands\Inventory\Health\Fix;

use App\Console\Commands\Inventory\Health\AbstractInventoryHealthCommandHelper;
use App\Console\Commands\Inventory\Health\Fix\Traits\FixSalesOrderLinesTrait;
use App\Managers\InventoryHealthManager;
use App\Models\SalesOrderLine;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Illuminate\Support\Collection;
use Throwable;
use function Laravel\Prompts\confirm;
use function Laravel\Prompts\table;
use function Laravel\Prompts\warning;

class FixMissingFulfillmentMovementsCommandHelper extends AbstractInventoryHealthCommandHelper
{
    use FixSalesOrderLinesTrait;

    public function getDescription(): string {
        return 'Sales order line fulfillments without negative reservation movements (Closed only)';
    }
    
    public function getQuery(): Builder
    {
        return $this->health->salesOrderLinesMissingFulfillmentMovementsQuery();
    }

    /** @var SalesOrderLine $dataItem */
    protected function getFixLabel(mixed $dataItem): string
    {
        return "Fixing fulfillment movements for sales order {$dataItem->salesOrder->sales_order_number} line $dataItem->id with SKU {$dataItem->product->sku} for quantity $dataItem->quantity";
    }

    /**
     * @throws Throwable
     * @var SalesOrderLine $dataItem
     */
    protected function fix($dataItem): void
    {
        try {
            $this->healthManager->createFulfillmentInventoryMovementsForSalesOrderLine($dataItem);
        } catch (ModelNotFoundException) {
            warning("Reservation movement also missing for sales order line $dataItem->id.  Fix that first before running the fulfillment movement fix");
            return;
        }
    }

    public function showProblemRecords(Collection $data): void
    {
        if (confirm("Show problem records?")) {
            $headers = [
                'Sales Order Line ID',
                'Sales Order ID',
                'Order #',
                'Qty',
                'Product ID',
                'SKU',
                'Warehouse',
                'Fulfillments',
            ];

            $rows = $data->map(function(SalesOrderLine $row) {
                $fulfillmentLines = $row->salesOrderFulfillmentLines->map(function($fulfillmentLine) {
                    return $fulfillmentLine->quantity . ' from ' . $fulfillmentLine->salesOrderFulfillment->fulfillment_number;
                })->implode(', ');
                return [
                    $row->id,
                    $row->sales_order_id,
                    $row->salesOrder->sales_order_number,
                    $row->quantity,
                    $row->product->id,
                    $row->product->sku,
                    $row->warehouse->name,
                    $fulfillmentLines
                ];
            })->toArray();

            table(
                headers: $headers,
                rows: $rows
            );

            $this->sendCsv($headers, $rows);
        }
    }
}