<?php

namespace App\Console\Commands\Inventory\Health;

use App\Managers\InventoryHealthManager;
use App\Notifications\ProblemRecordsCsvNotification;
use App\Repositories\InventoryHealthRepository;
use Illuminate\Console\Command;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Storage;
use Throwable;
use function Laravel\Prompts\info;
use function Laravel\Prompts\spin;
use function Laravel\Prompts\progress;
use function Laravel\Prompts\text;
use function Laravel\Prompts\warning;
use function Laravel\Prompts\confirm;

abstract class AbstractInventoryHealthCommandHelper
{
    protected InventoryHealthRepository $health;
    protected InventoryHealthManager $healthManager;
    protected Collection $data;
    protected Collection $warnings;
    protected int $numResults;
    
    public function __construct(protected Command $console)
    {
        $this->health = app(InventoryHealthRepository::class);
        $this->healthManager = app(InventoryHealthManager::class);
        $this->warnings = collect();
    }

    abstract public function getDescription(): string;
    abstract public function getQuery(): Builder;

    public function identify(): bool
    {
        info('Running query...');
        $this->data = $this->getQuery()->get();

        $numResults = spin(
            callback: fn() => $this->data->count(),
            message: 'Searching for ' . $this->getDescription() . '...',
        );

        info("Found $numResults " . $this->getDescription());
        if ($numResults == 0) {
            return false;
        }

        $this->showProblemRecords($this->data);

        return true;
    }

    /**
     * @throws Throwable
     */
    public function __invoke(): void
    {
        do {
            if (!$this->identify()) {
                return;
            }
            $this->filter();

            if ($this->data->isEmpty()) {
                info('No records to fix');
                return;
            }

            $progress = progress(label: 'Fixing ' . $this->getDescription() . '...', steps: $this->data->count());
            $progress->start();
            if ($this->allOrNothing()) {
                DB::beginTransaction();
            }
            foreach ($this->data as $dataItem)
            {
                $label = $this->getFixLabel($dataItem);
                $progress->label($label);
                customlog("inventory-fixes", $label, days: null);
                DB::transaction(fn() => $this->fix($dataItem));
                $progress->advance();
            }
            $progress->finish();
            $this->warnings->each(fn($warning) => warning($warning));
            $postFixSuccess = $this->postFix();
            if ($this->allOrNothing()) {
                DB::commit();
            }
        } while ($postFixSuccess);
    }

    abstract protected function fix(mixed $dataItem): void;
    abstract protected function filter(): void;
    abstract public function showProblemRecords(Collection $data): void;
    abstract protected function getFixLabel(mixed $dataItem): string;
    protected function postFix(): bool {
        return false;
    }
    protected function allOrNothing(): bool {
        return false;
    }

    public function sendCsv(array $headers, array $rows): void
    {
        if (
            confirm(
                label: "Email CSV of problem records?",
                default: false,
            )
        )
        {
            $toEmail = text('To what email address?');
            $filePath = Storage::disk('export')->path(uuid_create() . '.csv');

            $handle = fopen($filePath, 'w');
            fputcsv($handle, $headers);

            foreach($rows as $row) {
                fputcsv($handle, $row);
            }

            fclose ($handle);

            Notification::route('mail', $toEmail)->notify(new ProblemRecordsCsvNotification($filePath, count($rows) . ' ' . $this->getDescription()));
        }
    }

}