<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Redis;

class HorizonFlushCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'horizon:flush {--failed-only : Only delete failed jobs}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Flush all horizon data';

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $failedOnly = $this->option('failed-only');

        $this->info('Restarting queue');
        $this->call('queue:restart');

        $this->info('Terminating horizon');
        $this->call('horizon:terminate');

        // Delete failed_jobs table in mysql
        $this->info('Deleting failed_jobs table');
        $this->call('queue:flush');

        // I don't think this is safe to run if we are using cache for site functionality (like caching invalidated financials)
        //$this->call('cache:clear');

        // More severe options that I'm not sure are necessary
        // sudo killall -9 php8.1 && sudo supervisorctl restart all

        $redis = Redis::connection('horizon');
        $horizonPrefix = config('horizon.prefix');

        // Horizon prefix is already assumed for keys, so the match applies after the "horizon:" prefix
        $pattern = $failedOnly ? 'failed_jobs*' : '*';
        $keys = $redis->keys($pattern);

        $this->info(count($keys).' horizon entries found');

        if (count($keys) > 0) {
            $this->info('Flushing horizon entries');

            foreach ($keys as $key) {
                if ($this->output->isVerbose()) {
                    $this->info('Deleting key: '.$key);
                }
                $redis->del(str_replace($horizonPrefix, '', $key));
            }
        }

        $this->info('Terminating any rogue Horizon processes');
        $this->call('horizon:purge');

        /*
         * We are going to try to rely on horizon:terminate and monitor the behavior. Sometimes it triggers supervisor to restart horizon, other times it doesn't.
         * We do have a backup of the crontab jobs that we'll make check for horizon being down periodically.
         */
        //$this->info('Restarting supervisor');
        //shell_exec('sudo supervisorctl restart all');
    }
}
