<?php

namespace App\Console\Commands;

use App\Models\Address;
use App\Models\Addressable;
use App\Models\Customer;
use App\Models\PurchaseOrder;
use App\Models\SalesCredit;
use App\Models\SalesOrder;
use App\Models\Store;
use App\Models\Supplier;
use App\Models\Warehouse;
use Illuminate\Console\Command;

class DeleteDuplicateAddresses extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'patch:delete-duplicate-addresses';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Delete duplicate address records';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $uniqueFields = [
            'company',
            'name',
            'email',
            'address1',
            'address2',
            'address3',
            'city',
            'province',
            'province_code',
            'country',
            'country_code',
            'phone',
            'fax',
            'zip',
        ];

        $addressClassMappings = [
            Customer::class => [
                "default_shipping_address_id",
                "default_billing_address_id"
            ],
            Supplier::class => ["address_id"],
            Warehouse::class => ["address_id"],
            PurchaseOrder::class => ["destination_address_id"],
            Store::class => ["address_id"],
            SalesOrder::class => ["shipping_address_id", "billing_address_id"],
            SalesCredit::class => ["from_address_id"]
        ];

        $select = "";
        foreach ($uniqueFields as $field){
            $select .= "COALESCE($field, '') as $field,";
        }
        $select = rtrim($select, ',');

        $addressesWithDuplications = Address::query()
            ->selectRaw($select)
            ->groupBy('hash')
            ->havingRaw('COUNT(*) > 1')
            ->get();

        $addressesWithDuplications->each(function ($address) use ($addressClassMappings, $uniqueFields) {
            $addressUsagesQuery = Address::query();

            foreach ($uniqueFields as $field) {
                if (!empty($address->$field)) {
                    $addressUsagesQuery->where($field, $address->$field);
                } else {
                    $addressUsagesQuery->where(function ($query) use ($field) {
                        $query->whereNull($field)->orWhere($field, '')->orWhere($field, 0);
                    });
                }
            }

            $addressUsagesQuery->orderBy('id');

            $addressUsages = $addressUsagesQuery->get();

            if ($addressUsages->count() === 0) {
                $this->output->warning("No address usages found for address with the following query: " . $addressUsagesQuery->toSqlWithBindings());
                return false;
            }

            $addressIdToKeep = $addressUsages->first()->id;
            $addressIdsToDelete = $addressUsages->skip(1)->pluck('id');
            customlog("delete_duplicate_addresses", "Address ID to keep: {$addressIdToKeep}, Deleting " . $addressIdsToDelete->count() . " addresses");

            foreach ($addressClassMappings as $modelClass => $addressIdColumns) {
                foreach ($addressIdColumns as $addressIdColumn) {
                    app($modelClass)->whereIn($addressIdColumn, $addressIdsToDelete)->update([
                        $addressIdColumn => $addressIdToKeep
                    ]);
                }
            }

            Addressable::whereIn('address_id', $addressIdsToDelete)->delete();
            Address::whereIn('id', $addressIdsToDelete)->delete();
        });
    }
}
