<?php

namespace App\Console\Commands;

use App\Integrations\Starshipit;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\Starshipit\StarshipitOrder;
use App\Services\SalesOrder\FulfillSalesOrderService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CreateSkuFulfillmentFromStarShipIt extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'sku:create-sku-fulfillments-from-ssi
                            {ids : Specify order ids }';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Create sku fufillments from StarShipIt orders';

    /**
     * Create a new command instance.
     *
     * @return void
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     */
    public function handle(): void
    {
        $ids = explode(',', $this->argument('ids'));
        $order_numbers = [];

        $starshipitIntegrationInstance = IntegrationInstance::with([])->starshipit()->firstOrFail();

        $starshipit = new Starshipit($starshipitIntegrationInstance);

        foreach ($ids as $id) {
            $salesOrder = SalesOrder::with([])->findOrFail($id);
            echo $salesOrder->sales_order_number."\n";
            if ($salesOrder->fulfillment_status == SalesOrder::FULFILLMENT_STATUS_FULFILLED) {
                //Log::info("Skipping $salesOrder->sales_order_number since not fulfilled");
                continue;
            }
            $searchOrderResponse = $starshipit->searchOrders(['phrase' => $salesOrder->sales_order_number])->current();
            if ($searchOrderResponse->statusCode !== 200 || ! $searchOrderResponse->body['success']) {
                //Log::error('SSI search error', array_merge($searchOrderResponse->body, [$searchOrderResponse->statusCode]));
                continue;
            }
            //Log::info("Searching for $salesOrder->sales_order_number", $searchOrderResponse->body);
            if ($searchOrderResponse->body['orders']) {
                $order_numbers = collect($searchOrderResponse->body['orders'])->pluck('order_id');
            } else {
                $getOrdersResponse = $starshipit->getUnshippedOrders(['since_last_updated' => '2021-01-01T00:00:00.00Z', 'limit' => 250]);
                if ($getOrdersResponse->statusCode == 200 && $getOrdersResponse->body['success']) {
                    //Log::info('Getting all unshipped orders', $getOrdersResponse->body);
                    $order_numbers = collect($getOrdersResponse->body['orders'])->filter(function ($order) use ($salesOrder) {
                        return stristr($order['order_number'], $salesOrder->sales_order_number) !== false;
                    })->pluck('order_id');
                } else {
                    //Log::error('SSI get orders error', array_merge($getOrdersResponse->body, [$getOrdersResponse->statusCode]));
                }
            }

            if (! count($order_numbers)) {
                //Log::info("No SSI orders exist for $salesOrder->sales_order_number");
            }

            foreach ($order_numbers as $order_number) {
                $getOrderResponse = $starshipit->getOrder($order_number);

                if ($getOrderResponse->statusCode == 200 && $getOrderResponse->body['success']) {
                    /*
                     * Now we can use this payload to create the sales order fulfillment linked to starshipit
                     */

                    $ssiOrder = $getOrderResponse->body['order'];

                    $starshipitOrder = StarshipitOrder::with([])->firstOrNew(['order_id' => $ssiOrder['order_id']]);

                    $ssiLines = [];

                    foreach ($ssiOrder['items'] as $item) {
                        $matchingLine = $salesOrder->getMatchingUnfulfilledSalesOrderLineFromSku($item['sku']);
                        $warehouse_id = $matchingLine['warehouse_id'];
                        $ssiLines[] = [
                            'quantity' => $item['quantity'],
                            'sales_order_line_id' => $matchingLine['id'],
                        ];
                    }

                    $request = [
                        'fulfilled_at' => $ssiOrder['order_date'],
                        'fulfillment_lines' => $ssiLines,
                        'fulfillment_type' => 'starshipit',
                        'metadata' => json_encode(['signature_required' => $ssiOrder['signature_required']]),
                        'requested_shipping_method' => $ssiOrder['shipping_method'] ?? '',
                        'warehouse_id' => $warehouse_id,
                    ];

                    $fulfillment = FulfillSalesOrderService::make($salesOrder)->fulfillWithInputs($request, true, false);

                    if (! $starshipitOrder->exists) {
                        try {
                            $starshipitOrder->json_object = $ssiOrder;
                            $starshipitOrder->fill($ssiOrder);
                            $starshipitOrder->errors = null;
                            $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                            $starshipitOrder->save();
                        } catch (\Throwable $exception) {
                            //Log::debug("CreateSkuFulfillmentFromStarShipIt, order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                            try {
                                $starshipitOrder->rawOrder = $ssiOrder;
                                $starshipitOrder->errors = $exception->getMessage();
                                $starshipitOrder->save();
                            } catch (\Throwable $exception) {
                                //Log::debug("CreateSkuFulfillmentFromStarShipIt(saving raw), order_id:{$ssiOrder['order_id']}: {$exception->getMessage()}", $exception->getTrace());
                            }
                        }
                    } else {
                        //Log::info('SSI order '.$ssiOrder['order_id'].' already exists, but creating sku fulfillments');
                        $starshipitOrder->sku_fulfillment_id = $fulfillment->id;
                        $starshipitOrder->save();
                    }

                    //Log::info("Imported fulfillment for $salesOrder->sales_order_number (".$ssiOrder['order_id'].')', $request);
                }
            }
        }
    }
}
