<?php

namespace App\Actions;

use App\Data\CreateInitialStockTakeData;
use App\Data\CreateInitialStockTakeItemData;
use App\Data\InitialInventoryData;
use App\Data\StockTakeItemData;
use App\Data\UpdateStockTakeData;
use App\Helpers;
use App\Models\Product;
use App\Models\Setting;
use App\Repositories\StockTakeRepository;
use App\Services\StockTake\StockTakeManager;
use Carbon\Carbon;
use DB;
use Throwable;

class SetInitialInventory
{
    private StockTakeRepository $stockTakes;
    private StockTakeManager $stockTakeManager;

    public function __construct(private readonly Product $product, private readonly InitialInventoryData $data)
    {
        $this->stockTakes = app(StockTakeRepository::class);
        $this->stockTakeManager = app(StockTakeManager::class);
    }

    /**
     * @throws Throwable
     */
    public function handle(): void
    {
        DB::transaction(function ()
        {
            if ($stockTake = $this->stockTakes->getInitialStockTakeForWarehouseId($this->data->warehouse_id))
            {
                $this->stockTakeManager->modifyStockTake($stockTake, UpdateStockTakeData::from([
                    'items' => StockTakeItemData::collection([[
                        'product_id' => $this->product->id,
                        'qty_counted' => $this->data->quantity,
                        'unit_cost' => $this->data->unit_cost,
                        'to_delete' => $this->data->quantity == 0,
                    ]])
                ]));
            } else {
                $this->stockTakeManager->createInitialStockTake(CreateInitialStockTakeData::from([
                    'date_count' => Carbon::parse(Helpers::setting(Setting::KEY_INVENTORY_START_DATE, useCache: true)),
                    'warehouse_id' => $this->data->warehouse_id,
                    'items' => CreateInitialStockTakeItemData::collection([CreateInitialStockTakeItemData::from([
                        'product_id' => $this->product->id,
                        'qty_counted' => $this->data->quantity,
                        'unit_cost' => $this->data->unit_cost,
                    ])])
                ]));
            }
        });
    }
}