<?php

namespace App\Actions\InventoryManagement;

use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\InvalidQuantityForSplittingInventoryMovementException;
use App\Exceptions\InventoryMovementTypeException;
use App\Exceptions\SupplierWarehouseCantHaveInventoryMovementsException;
use App\Models\InventoryMovement;
use App\Models\SalesOrderLine;
use App\Models\WarehouseTransferShipmentLine;
use App\Repositories\InventoryMovementRepository;
use App\Services\StockTake\OpenStockTakeException;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Throwable;

class SplitInventoryMovement
{
    public function __construct(
        private readonly InventoryMovementRepository $movements
    )
    {
    }

    /**
     * @throws InsufficientSalesOrderLineQuantityException
     * @throws InvalidQuantityForSplittingInventoryMovementException
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     */
    public function __invoke(
        InventoryMovement $inventoryMovement,
        int $quantityToSplit,
    ): InventoryMovement
    {
        if (abs($quantityToSplit) >= abs($inventoryMovement->quantity)) {
            throw new InvalidQuantityForSplittingInventoryMovementException("Quantity to split $quantityToSplit cannot be greater than the inventory movement quantity $inventoryMovement->quantity");
        }
        if ($inventoryMovement->quantity >= 0 || $inventoryMovement->inventory_status !== InventoryMovement::INVENTORY_STATUS_ACTIVE) {
            throw new InvalidQuantityForSplittingInventoryMovementException("Inventory movement quantity must be negative and active to split");
        }

        $this->handleSalesOrderLineLinkType($inventoryMovement, $quantityToSplit);
        $this->handleWarehouseTransferShipmentLineLinkType($inventoryMovement, $quantityToSplit);

        $newInventoryMovement = $inventoryMovement->replicate();
        $newInventoryMovement->quantity = -abs($quantityToSplit);
        $newInventoryMovement->save();

        $inventoryMovement->quantity = -(abs($inventoryMovement->quantity) - abs($quantityToSplit));
        $inventoryMovement->save();

        return $newInventoryMovement;
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function handleSalesOrderLineLinkType(InventoryMovement $activeMovement, int $quantityToSplit): void
    {
        if ($activeMovement->link_type !== SalesOrderLine::class) {
            return;
        }
        // Get corresponding reservation and fulfillment and move those too
        $reservationMovement = $this->movements->getCorrespondingReservationMovementForActiveSalesOrderLineMovement($activeMovement);
        if (!$reservationMovement) {
            customlog('inventory-fixes', "No corresponding reservation movement found for active sales order line movement ID $activeMovement->id", days: null);
//            dd("Can't find reservation movement when trying to split inventory movement id $activeMovement->id", $activeMovement->toArray(),
//                InventoryMovement::query()
//                    ->where('link_id', $activeMovement->link_id)
//                    ->where('link_type', SalesOrderLine::class)
//                    ->get()->toArray());
            return;
        }

        $fulfillmentMovement = $this->movements->getCorrespondingFulfillmentMovementForReservationSalesOrderLineMovement($reservationMovement);

        $newReservationMovement = $reservationMovement->replicate();
        $newReservationMovement->quantity = abs($quantityToSplit);
        $newReservationMovement->save();

        $reservationMovement->quantity = abs($reservationMovement->quantity) - abs($quantityToSplit);
        $reservationMovement->save();

        // TODO: I'm unclear on how the fulfillment movements should go if there are more than 1.  Needs a unit test
        //  for now just handling the single and erroring using sole if there are more than 1
        if ($fulfillmentMovement) {
            $newFulfillmentMovement = $fulfillmentMovement->replicate();
            $newFulfillmentMovement->quantity = - abs($quantityToSplit);
            $newFulfillmentMovement->save();

            $fulfillmentMovement->quantity = -(abs($fulfillmentMovement->quantity) - abs($quantityToSplit));
            $fulfillmentMovement->save();
        }
    }

    /**
     * @throws InventoryMovementTypeException
     * @throws OpenStockTakeException
     * @throws Throwable
     * @throws SupplierWarehouseCantHaveInventoryMovementsException
     */
    private function handleWarehouseTransferShipmentLineLinkType(
        InventoryMovement $movement,
        int $quantityToSplit
    ): void {
        if ($movement->link_type !== WarehouseTransferShipmentLine::class) {
            return;
        }

        $addInTransitMovement = $this->movements->getCorrespondingAddInTransitMovementForActiveLineMovement($movement);
        $deductInTransitMovements = $this->movements->getCorrespondingDeductInTransitMovementForAddInTransitLineMovement($addInTransitMovement);

        $newAddInTransitMovement = $addInTransitMovement->replicate();
        $newAddInTransitMovement->quantity = abs($quantityToSplit);
        $newAddInTransitMovement->save();

        $addInTransitMovement->quantity = abs($addInTransitMovement->quantity) - abs($quantityToSplit);
        $addInTransitMovement->save();

        $deductInTransitMovements->each(function(InventoryMovement $deductInTransitMovement) use ($quantityToSplit) {

            $newDeductInTransitMovement = $deductInTransitMovement->replicate();
            $newDeductInTransitMovement->quantity = -abs($quantityToSplit);
            $newDeductInTransitMovement->save();

            $deductInTransitMovement->quantity = -(abs($deductInTransitMovement->quantity) - abs($quantityToSplit));
            $deductInTransitMovement->save();
        });

        $newActiveMovement = $movement->replicate();
        $newActiveMovement->quantity = -abs($quantityToSplit);
        $newActiveMovement->save();
    }
}
