<?php

namespace App\Actions\InventoryManagement;

use App\Data\CreateBackorderQueueData;
use App\Data\FifoLayerAllocationData;
use App\Data\FifoLayerQuantityData;
use App\Exceptions\CantBackorderAlreadyFulfilledQuantityException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Managers\InventoryMovementManager;
use App\Models\BackorderQueue;
use App\Models\SalesOrderLine;
use App\Repositories\FifoLayerRepository;
use App\Repositories\SalesOrderLineRepository;
use App\Services\InventoryManagement\BackorderManager;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

class CreateSalesOrderReservation
{
    protected SalesOrderLine $salesOrderLine;
    protected FifoLayerAllocationData $layerData;
    protected ?BackorderQueue $backorder;
    protected Collection $inventoryMovementsCreated;

    public function __construct(
        private readonly GetFifoLayersForQuantity $getFifoLayersForQuantity,
        private readonly BackorderManager $backorderManager,
        private readonly InventoryMovementManager $inventoryMovementManager,
        private readonly FifoLayerRepository $fifoLayers,
        private readonly SalesOrderLineRepository $salesOrderLines,
    )
    {
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     * @throws Throwable
     */
    public function __invoke(
        SalesOrderLine $salesOrderLine
    ): Collection
    {
        $this->inventoryMovementsCreated = collect();
        $this->salesOrderLine = $salesOrderLine;
        $this->backorder = null;
        $this->layerData = ($this->getFifoLayersForQuantity)(
            $salesOrderLine->product,
            $salesOrderLine->warehouse,
            $salesOrderLine->quantity - $salesOrderLine->externally_fulfilled_quantity,
            true,
        );

        $backorderQuantity = $this->layerData->remainingQuantityToAllocate;
        if ($backorderQuantity > 0) {
            $unfulfilledQuantity = $this->salesOrderLines->calculateUnfulfilledQuantity($salesOrderLine);
            // You can only backorder unfulfilled quantity, so any backorder quantity over the unfulfilled quantity is an adjustment
            $adjustmentQuantityNeeded = $backorderQuantity - $unfulfilledQuantity;
            if ($adjustmentQuantityNeeded > 0) {
                throw new CantBackorderAlreadyFulfilledQuantityException(
                    "Sales Order Line ID {$this->salesOrderLine->id}, {$this->salesOrderLine->product->sku} (ID: {$this->salesOrderLine->product_id}) ({$this->salesOrderLine->salesOrder->sales_order_number}) can't be backordered for already fulfilled quantity. Need a positive inventory adjustment of $adjustmentQuantityNeeded to reflect the historical fulfillment."
                );
            }
            $this->backorder = $this->backorderManager->newCreateBackorder(CreateBackorderQueueData::from([
                'salesOrderLine' => $salesOrderLine,
                'backordered_quantity' => $backorderQuantity,
            ]));
        }

        DB::transaction(function () {
            $this->createInventoryMovements();
            $this->updateFifoLayerCache();
        });

        return $this->inventoryMovementsCreated;
    }

    /**
     * @throws Throwable
     */
    private function createInventoryMovements(): void
    {
        $this->layerData->fifoLayerQuantities->each(function (FifoLayerQuantityData $layerData) {
            $this->inventoryMovementsCreated = $this->inventoryMovementsCreated->merge($this->inventoryMovementManager->createSalesOrderReservationMovementsForFifo(
                $this->salesOrderLine,
                $layerData
            ));
        });
        if ($backorder = $this->backorder) {
            $this->inventoryMovementsCreated = $this->inventoryMovementsCreated->merge($this->inventoryMovementManager->createSalesOrderReservationMovementsForBackorder(
                $this->salesOrderLine,
                $backorder
            ));
        }
    }

    private function updateFifoLayerCache(): void
    {
        $this->layerData->fifoLayerQuantities->each(function (FifoLayerQuantityData $fifoLayerQuantityData) {
            $this->fifoLayers->validateFifoLayerCache($fifoLayerQuantityData->fifoLayer);
        });
    }
}
