<?php

namespace App\Actions\InventoryHealth;

use App\Models\FifoLayer;
use App\Models\SalesOrderLine;
use App\Models\SalesOrderLineLayer;
use App\Services\FinancialManagement\SalesOrderLineFinancialManager;
use Exception;
use Illuminate\Database\MultipleRecordsFoundException;

class FixSalesOrderLineLayerCache
{
    protected SalesOrderLine $salesOrderLine;

    public function __construct(
        private readonly SalesOrderLineFinancialManager $salesOrderLineFinancialManager,
    )
    {
    }

    /**
     * @throws Exception
     */
    public function __invoke(
        SalesOrderLine $salesOrderLine,
    ): void
    {
        $this->salesOrderLine = $salesOrderLine;
        $this->createLayer();
        $this->updateCache();

    }

    private function createLayer(): void
    {
        try {
            $activeMovement = $this->salesOrderLine->getActiveInventoryMovement();
        } catch (MultipleRecordsFoundException) {
            customlog('inventory-fixes', 'Multiple active inventory movements found for sales order line', [
                'sales_order_line_id' => $this->salesOrderLine->id,
            ], days: null);
            return;
        }
        SalesOrderLineLayer::create([
            'sales_order_line_id' => $this->salesOrderLine->id,
            'quantity' => abs($activeMovement->quantity),
            'layer_id' => $activeMovement->layer->id,
            'layer_type' => FifoLayer::class,
        ]);
    }

    /**
     * @throws Exception
     */
    private function updateCache(): void
    {
        $this->salesOrderLineFinancialManager->calculateAndCacheFinancials(collect([$this->salesOrderLine]));
        $this->salesOrderLine->updated_at = now();
        $this->salesOrderLine->save();
    }
}
