<?php

namespace App\Actions\InventoryHealth;

use App\Exceptions\InsufficientSalesOrderLineQuantityException;
use App\Exceptions\OversubscribedFifoLayerException;
use App\Exceptions\UnableToAllocateToFifoLayersException;
use App\Models\FifoLayer;
use App\Models\InventoryMovement;
use App\Repositories\FifoLayerRepository;
use Throwable;

class FixOversubscribedFifoLayer
{
    protected FifoLayer $fifoLayer;
    protected int $overSubscribedQty;

    public function __construct(
        private readonly FifoLayerInventoryMovementRelocator $fifoLayerInventoryMovementRelocator,
        private readonly FifoLayerRepository $layers,
        private readonly BackorderFromFifoLayer $backorderFromFifoLayer,
    )
    {
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws UnableToAllocateToFifoLayersException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    public function __invoke(
        FifoLayer $fifoLayer,
    ): int
    {
        $this->fifoLayer = $fifoLayer;

        $this->validateOverSubscription();

        if ($this->overSubscribedQty <= 0) {
            return 0;
        }

        $this->fixOverSubscribedFifoLayer();
        $this->layers->validateFifoLayerCache($fifoLayer);

        if ($this->overSubscribedQty > 0) {
            customlog('inventory-fixes', "Fifo Layer {$this->fifoLayer->id} still has $this->overSubscribedQty oversubscribed", days: null);
        }

        return $this->overSubscribedQty;
    }

    private function validateOverSubscription(): void
    {
        $this->overSubscribedQty = abs($this->fifoLayer->inventoryMovements->where('inventory_status', InventoryMovement::INVENTORY_STATUS_ACTIVE)->sum('quantity'));
    }

    /**
     * @throws UnableToAllocateToFifoLayersException
     * @throws InsufficientSalesOrderLineQuantityException
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     */
    private function fixOverSubscribedFifoLayer(): void
    {
        $this->relocate();
        $this->backorder();
    }

    /**
     * @throws OversubscribedFifoLayerException
     * @throws Throwable
     * @throws InsufficientSalesOrderLineQuantityException
     */
    private function relocate(): void
    {
        $fifoAllocationData = ($this->fifoLayerInventoryMovementRelocator)($this->overSubscribedQty, $this->fifoLayer);
        $this->overSubscribedQty = $fifoAllocationData->remainingQuantityToAllocate;
        $this->fifoLayer->refresh();
    }

    private function backorder(): void
    {
        if ($this->overSubscribedQty > 0) {
            $qtyBackordered = ($this->backorderFromFifoLayer)($this->fifoLayer, $this->overSubscribedQty);
            $this->overSubscribedQty -= $qtyBackordered;
        }
    }
}
