<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\AbstractIntegrationRepository;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Models\Product;
use App\Models\ProductListing;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Throwable;

abstract class AbstractSalesChannelProductRepository extends AbstractIntegrationRepository implements SalesChannelProductRepositoryInterface
{
    /**
     * @throws Throwable
     */
    public function bulkDelete(array $ids): void
    {
        foreach (array_chunk($ids, 250) as $chunk)
        {
            DB::transaction(function () use ($chunk)
            {
                $salesChannelProductsQuery = $this->model
                    ->with('productListing')
                    ->whereIn('id', $chunk);

                $salesChannelProducts = $salesChannelProductsQuery->get();

                // Delete product listings
                $productListingIds = $salesChannelProducts
                    ->filter(function ($salesChannelProduct) {
                        return $salesChannelProduct->productListing;
                    })
                    ->pluck('productListing.id');

                ProductListing::whereIn('id', $productListingIds)->delete();

                $salesChannelProductsQuery->delete();
            });
        }
    }

    public function getStartDateForNew(IntegrationInstanceInterface $integrationInstance): ?Carbon
    {
        if (!$this->model::getLastModified()) {
            return null;
        }

        $startDate = $this->model::where('integration_instance_id', $integrationInstance->id)
            ->latest($this->model::getLastModified())
            ->pluck($this->model::getLastModified())
            ->first();

        return $startDate ? Carbon::parse($startDate)->addSecond() : null;
    }

    public function getProductFromId(IntegrationInstanceInterface $integrationInstance, string $productId): ?AbstractSalesChannelProduct
    {
        return $this->model::query()
            ->where([
                'integration_instance_id' => $integrationInstance->id,
                $this->model::getUniqueField() => $productId,
            ])->first();
    }

    public function getProductsFromUniqueSalesChannelProductIds(array $uniqueIds, AbstractSalesChannelProduct $model, IntegrationInstanceInterface $integrationInstance): EloquentCollection
    {
        $chunkSize = 10000;
        $products = new EloquentCollection();

        foreach (array_chunk($uniqueIds, $chunkSize) as $chunk) {
            $productsChunk = $model::query()
                ->where('integration_instance_id', $integrationInstance->id)
                ->whereIn($model::getUniqueField(), $chunk)
                ->get();

            $products = $products->merge($productsChunk);
        }

        return $products;
    }

    public function getSkuProductFromUniqueSalesChannelProductId(string $uniqueId, IntegrationInstanceInterface $integrationInstance): ?Product
    {
        return ProductListing::query()
            ->where('sales_channel_id', $integrationInstance->salesChannel->id)
            ->where('sales_channel_listing_id', $uniqueId)
            ->first()?->product;
    }

    public function getProductListingsForInventorySync(?array $ids = []): Collection
    {
        $listings = ProductListing::query()
            ->where(function (Builder $query) {
                $query->where('sales_channel_qty', null);
                $query->orWhereColumn('sales_channel_qty', '!=', 'quantity');
            });

        if(!empty($ids)) {
            $listings->whereHas('document', function (Builder $query) use ($ids) {
                $query->whereIn($this->model->getTable() . '.id', $ids);
            });
        }

        return $listings->get();
    }
}