<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Enums\OrderSyncStatusEnum;
use App\Repositories\SalesOrder\SalesOrderRepository;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Modules\Amazon\Enums\Entities\OrderStatusEnum;
use Throwable;

abstract class AbstractSalesChannelOrderRepository implements SalesChannelOrderRepositoryInterface
{
    abstract protected static function getModelClassName(): string;

    abstract protected static function getLinesModelClassName(): string;

    protected SalesChannelOrderInterface $model;

    protected SalesChannelOrderLineInterface $modelLine;

    public function __construct()
    {
        $this->model = app(static::getModelClassName());
        $this->modelLine = app(static::getLinesModelClassName());
    }

    public function save(IntegrationInstanceInterface $integrationInstance, Collection $orders): void
    {
        $upsertData = $orders->map(function ($order) use ($integrationInstance) {
            return [
                'integration_instance_id' => $integrationInstance->id,
                'error_log' => $this->modelLine::getLineItemsKey() ? null : OrderSyncStatusEnum::PENDING_ITEMS,
                'json_object' => json_encode($order->json_object),
            ];
        })->toArray();

        $this->model::upsert($upsertData, ['integration_instance_id', $this->model::getTableUniqueId()]);

        if ($this->modelLine::getLineItemsKey())
        {
            $this->saveLines($integrationInstance, $orders);
        }
    }

    private function saveLines(IntegrationInstanceInterface $integrationInstance, Collection $orders): void
    {
        $salesChannelOrders = $this->model::where('integration_instance_id', $integrationInstance->id)
            ->whereIn($this->model::getTableUniqueId(), $orders->pluck('json_object.' . $this->model::getUniqueId()))
            ->get();

        $lineItemsCollection = [];
        $salesChannelOrders->each(function (AbstractSalesChannelOrder $salesChannelOrder) use ($orders, &$lineItemsCollection)
        {
            $lineItems = $orders
                ->where('json_object.' . $this->model::getUniqueId(), $salesChannelOrder->json_object[$this->model::getUniqueId()])
                ->first()
                ->json_object[$this->modelLine::getLineItemsKey()];
            $lineItems = collect($lineItems)->map(function ($lineItem) use ($salesChannelOrder) {
                return [
                    $this->modelLine::getParentRelationId() => $salesChannelOrder->id,
                    'json_object' => json_encode($lineItem),
                ];
            })->toArray();

            array_push($lineItemsCollection, ...$lineItems);
        });

        $this->modelLine::upsert($lineItemsCollection, ['integration_instance_id', $this->modelLine::getTableUniqueId()]);
    }

    /**
     * @throws Throwable
     */
    public function bulkDelete(array $ids): void
    {
        foreach (array_chunk($ids, 250) as $chunk)
        {
            DB::transaction(function () use ($chunk)
            {
                $salesChannelOrdersQuery = $this->model
                    ->with('salesOrder')
                    ->whereIn('id', $chunk);

                $salesChannelOrders = $salesChannelOrdersQuery->get();

                // Delete sales orders
                $salesOrderIds = $salesChannelOrders
                    ->filter(function ($salesChannelOrder) {
                        return $salesChannelOrder->salesOrder;
                    })
                    ->pluck('salesOrder.id');
                app(SalesOrderRepository::class)->bulkDelete($salesOrderIds->toArray());

                $this->modelLine::whereIn($this->modelLine->getParentRelationId(), $chunk)->delete();

                $salesChannelOrdersQuery->delete();
            });
        }
    }

    public function getStartDateForNew(IntegrationInstanceInterface $integrationInstance): Carbon
    {
        $startDate = $this->model::where('integration_instance_id', $integrationInstance->id)
            ->latest($this->model::getLastModified())
            ->pluck($this->model::getLastModified())
            ->first();

        if ($startDate) {
            $date = Carbon::parse($startDate)->addSecond();
        } else {
            $date = Carbon::parse($integrationInstance->integration_settings['start_date']);
        }

        return $date;
    }

    public function getOrderFromId(IntegrationInstanceInterface $integrationInstance, string $orderId): ?AbstractSalesChannelOrder
    {
        return $this->model::where([
                'integration_instance_id' => $integrationInstance->id,
                $this->model->getTableUniqueId() => $orderId,
            ])
            ->first();
    }

    /**
     * @return Collection<array-key, Builder>
     */
    public function getCanceledOrdersQuery(IntegrationInstanceInterface $integrationInstance, ?Collection $collection = null): Builder
    {
        $query = $this->model::query()
            ->with('salesOrder')
            ->whereHas('salesOrder', function ($query) {
                $query->whereNull('canceled_at'); //We don't to cancel orders which are already canceled
            })
            ->where('integration_instance_id', $integrationInstance->id);

        if ($collection && $collection->isNotEmpty()) {
            $query->whereIn($this->model::getTableUniqueId(), $collection->pluck($this->model::getUniqueId()));
        }

        return $query;
    }

    abstract public function getCanceledOrders(IntegrationInstanceInterface $integrationInstance, ?Collection $collection = null): Collection;
}
