<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Helpers;
use App\Models\Concerns\BulkImport;
use App\Models\IntegrationInstance;
use App\Models\ProductListing;
use App\Models\SalesOrderLine;
use App\Models\Setting;
use Awobaz\Compoships\Compoships;
use Illuminate\Contracts\Database\Eloquent\Builder as EloquentBuilder;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;

abstract class AbstractSalesChannelOrderLine extends Model implements SalesChannelOrderLineInterface
{
    use BulkImport,
        Compoships;

    protected $guarded = [];

    protected $casts = [
        'json_object' => 'array',
    ];

    const BULK_THRESHOLD = 10;

    public function salesOrderLine(): HasOne
    {
        return $this->hasOne(SalesOrderLine::class,
            'sales_channel_line_id',
            static::getTableUniqueId(),
        )->whereHas('salesOrder', function (Builder $builder) {
            $builder->whereHas('salesChannel', function (Builder $builder) {
                $builder->where('integration_instance_id', $this->order->integration_instance_id);
            });
        });
    }

    public function getWarehouseId(): ?int
    {
        if (! $this->product) {
            return null;
        }

        return $this->order->warehouseRepository->getPriorityWarehouseIdForProduct(
            $this->product?->product,
            $this->quantity,
        );
    }

    public function getNominalCodeId(): ?int
    {
        return Helpers::setting(Setting::KEY_NC_MAPPING_SALES_ORDERS);
    }

    // Not common to implement
    public function getDiscountAllocation(): float
    {
        return 0;
    }

    public function scopeWithProduct(EloquentBuilder $builder, IntegrationInstanceInterface $integrationInstance)
    {
        $salesChannelProductClass = $this->salesChannelProductClass();
        $uniqueSalesChannelProductColumn = (new $salesChannelProductClass())::getUniqueField();

        $builder
            ->addSelect($this->getTable().'.*')
            ->addSelect('product_listings.id as productListingsId')
            ->addSelect('product_listings.product_id as productListingsProductId')
            ->leftJoin((new $salesChannelProductClass())->getTable().' as sales_channel_product_table', function ($query) use ($integrationInstance, $uniqueSalesChannelProductColumn) {
                $query->on('sales_channel_product_table.'.$uniqueSalesChannelProductColumn, $this->getTable().'.'.static::getSalesChannelProductUniqueId())
                    ->where('integration_instance_id', $integrationInstance->id);
            })
            ->leftJoin((new ProductListing())->getTable(), function ($join) use ($salesChannelProductClass) {
                $join->on((new ProductListing())->getTable().'.document_id', 'sales_channel_product_table.id')
                    ->where((new ProductListing())->getTable().'.document_type', $salesChannelProductClass);
            });

        return $builder;
    }
}
