<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Data\CreateSkuOrderFromSalesChannelData;
use App\DataTable\DataTable;
use App\Http\Controllers\Controller;
use App\Response;
use Exception;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Http\Request;
use Spatie\LaravelData\Optional;
use Throwable;

abstract class AbstractSalesChannelOrderController extends Controller
{
    use DataTable;

    abstract protected function getModel(): string;

    abstract protected function getResource(): string;

    public function abstractShow(SalesChannelOrderInterface $order)
    {
        $order->load('orderItems');
        return $this->response->addData($this->getResource()::make($order));
    }

    /**
     * @throws Exception|Throwable
     */
    public function abstractCreateSkuOrders(CreateSkuOrderFromSalesChannelData $data, IntegrationInstanceInterface $integrationInstance): Response
    {
        //Create an instance for order manager
        $manager = app($this->getModel())->manager($integrationInstance);

        if (! $data->filters instanceof Optional) {
            $request = new Request($data->toArray());
            $data->ids = $this->getIdsFromFilters($this->getModel(), $request, $integrationInstance);
        }

        //Create SKU orders
        $manager->createSkuOrders((@$data->ids ?: []), $data->create_all_orders);

        return $this->response->setMessage('Sku.io sales orders successfully created');
    }

    public function abstractRefresh(ShouldQueue $refreshJob): Response
    {
        dispatch($refreshJob)->onQueue('sales-channels');

        return $this->response->setMessage('Added to queue, orders will be refreshed shortly.');
    }

    public function abstractRefreshOrder(Request $request, IntegrationInstanceInterface $integrationInstance): Response
    {
        $request->validate([
            'uniqueId' => 'required|string',
        ]);
        $order = app($this->getModel())->manager($integrationInstance)->getOrder($request->input('uniqueId'));

        return $this->response->addData($this->getResource()::make($order));
    }
}