<?php

namespace App\Abstractions\Integrations\SalesChannels;

use App\Collections\SalesOrderLineCollection;
use App\Data\FinancialLineData;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\DTO\FieldValueMappingDto;
use App\Enums\FinancialLineClassificationEnum;
use App\Enums\FinancialLineProrationStrategyEnum;
use App\Helpers;
use App\Models\Concerns\Archive;
use App\Models\Concerns\BulkImport;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Currency;
use App\Models\IntegrationInstance;
use App\Models\SalesOrder;
use App\Models\Setting;
use App\Repositories\FinancialLineRepository;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Illuminate\Support\Arr;
use Spatie\LaravelData\DataCollection;
use Throwable;

/**
 *
 * @property SalesOrder $salesOrder
 * @property EloquentCollection $orderItems
 */
abstract class AbstractSalesChannelOrder extends Model implements DataTableModelInterface, SalesChannelOrderInterface
{
    use Archive;
    use BulkImport;
    use DataTableModelTrait;
    use HasFilters;
    use HasSort;

    protected $guarded = [];

    const BULK_THRESHOLD = 50;

    abstract public static function newFactory(): Factory;

    abstract public static function getUniqueId(): string;

    abstract public static function getIntegrationName(): string;

    abstract public static function getItemClassName(): string;

    public function getIsFulfillable(): bool
    {
        return true;
    }

    public static function getDateRestrictionExceptionMapping(): ?FieldValueMappingDto
    {
        return null;
    }

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(IntegrationInstance::class);
    }

    abstract public function orderItems(): HasMany;

    public function salesOrder(): MorphOne
    {
        return $this->morphOne(SalesOrder::class, 'sales_channel_order');
    }

    public function getShippingLine(): ?FinancialLineData
    {
        if ($this->getShippingTotal() > 0) {
            $financialLineType = app(FinancialLineRepository::class)
                ->getOrCreateFinancialLineType('Shipping', FinancialLineClassificationEnum::REVENUE, Helpers::setting(Setting::KEY_NC_MAPPING_SHIPPING_SALES_ORDERS));

            return FinancialLineData::from([
                'sales_order_number' => $this->{static::getUniqueId()},
                'financial_line_type_id' => $financialLineType->id,
                'description' => 'Shipping',
                'quantity' => 1,
                'amount' => $this->getShippingTotal(),
                'tax_allocation' => $this->getShippingTaxTotal(),
                'allocate_to_products' => $financialLineType->allocate_to_products ?? 1,
                'proration_strategy' => $financialLineType->proration_strategy ?? FinancialLineProrationStrategyEnum::REVENUE_BASED,
                'nominal_code_id' => $financialLineType->nominal_code_id ?? $this->integrationInstance->getSalesNominalCodeId(),
            ]);
        }

        return null;
    }

    // Discount lines are not common so putting these defaults in the abstraction

    public function getDiscountTotal(): float
    {
        return 0;
    }

    public function getDiscountTaxTotal(): float
    {
        return 0;
    }

    public function getDiscountLine(): ?FinancialLineData
    {
        if ($this->getDiscountTotal() == 0) {
            return null;
        }

        $financialLineType = app(FinancialLineRepository::class)
            ->getOrCreateFinancialLineType('Discount', FinancialLineClassificationEnum::COST);

        return FinancialLineData::from([
            'sales_order_id' => $this->id,
            'financial_line_type_id' => $financialLineType->id,
            'description' => 'Discount',
            'quantity' => 1,
            'amount' => $this->getDiscountTotal(),
            'tax_allocation' => $this->getDiscountTaxTotal(),
            'allocate_to_products' => $financialLineType->allocate_to_products ?? 1,
            'proration_strategy' => $financialLineType->proration_strategy ?? FinancialLineProrationStrategyEnum::REVENUE_BASED,
        ]);
    }

    public function getSalesOrderLines(): SalesOrderLineCollection
    {
        $salesOrderLines = new SalesOrderLineCollection();

        $this->orderItems->each(function (SalesChannelOrderLineInterface $orderItem) use (&$salesOrderLines) {
            $salesOrderLines->push($orderItem->getSalesOrderLineData());
        });

        return $salesOrderLines;
    }

    public function getFinancialLines(): DataCollection
    {
        $financialLines = FinancialLineData::collection([]);

        // Add shipping line
        $shippingLine = $this->getShippingLine();
        if ($shippingLine) {
            $financialLines[] = $this->getShippingLine();
        }

        //Add discount line
        $discountLine = $this->getDiscountLine();
        if ($discountLine) {
            $financialLines[] = $this->getDiscountLine();
        }

        return $financialLines;
    }

    /**
     * @throws Throwable
     */
    public function getCurrencyCode(?string $salesChannelOrderCurrencyCode = null): string
    {
        $currencyCode = $salesChannelOrderCurrencyCode ?: Currency::default()->code;
        throw_if(
            Currency::query()->where('code', $currencyCode)
                ->doesntExist(),
            'Currency code '.$currencyCode.' does not exist in the system.'
        );

        return $currencyCode;
    }

    public function availableColumns(): array
    {
        $columns = array_merge(Arr::except($this->getAllColumnsAndTypes(), [
            'integration_instance_id',
            'json_object',
        ]));

        $firstTwo = array_slice($columns, 0, 2);
        $rest = array_slice($columns, 2);
        $columns = array_merge($firstTwo, [
            'sku_sales_order_id' => 'integer',
        ], $rest);

        $this->availableColumns = $columns;

        return $this->availableColumns;
    }

    public function generalFilterableColumns(): array
    {
        return ['id', static::getUniqueId()];
    }

    /*
    |--------------------------------------------------------------------------
    | Scopes
    |--------------------------------------------------------------------------
    */

    /*
     * If the Sales Channel Order has a currency field name defined:
     * Scope a query to only include orders that have a currency that exists in the currencies table
     * If the currency value is empty or null, we will assume the default currency anyway so don't filter the order out
     */
    public function scopeCurrencyExists(Builder $builder): Builder
    {
        if (is_null(static::getCurrency())) {
            return $builder;
        }

        $builder->where(function (Builder $builder) {
            $builder->whereIn(static::getCurrency(), function ($builder) {
                $builder->from('currencies')->select('code');
            });
            $builder->orWhere(static::getCurrency(), '');
            $builder->orWhereNull(static::getCurrency());
        });

        return $builder;
    }

    public function scopeDateRestrictionException(Builder $builder): Builder
    {
        if (! (static::getDateRestrictionExceptionMapping())) {
            return $builder;
        }

        return $builder->orWhere(static::getDateRestrictionExceptionMapping()->field, static::getDateRestrictionExceptionMapping()->value);
    }
}
