<?php

namespace App\Abstractions\Integrations\Accounting;

use App\Abstractions\Integrations\AbstractAccountingIntegrationManager;
use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Models\AccountingTransaction;
use App\Models\InventoryAdjustment;
use App\Models\Payment;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderShipmentReceipt;
use App\Models\SalesCredit;
use App\Models\SalesCreditReturn;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Setting;
use App\Models\StockTake;
use App\Models\Supplier;
use App\Repositories\NominalCodeRepository;
use App\Repositories\SettingRepository;
use App\Services\Amazon\FBAInventory\CustomerReturn;
use Illuminate\Support\Collection;
use Illuminate\Support\LazyCollection;
use Modules\Qbo\ApiParameterObjects\QboCreateCustomerApo;
use Modules\Qbo\ApiParameterObjects\QboCreateItemApo;
use Modules\Qbo\ApiParameterObjects\QboCreateVendorApo;
use Modules\Qbo\ApiParameterObjects\QboUpdateOrCreateCreditMemosApo;
use Modules\Qbo\ApiParameterObjects\QboUpdateOrCreateInvoicesApo;
use Modules\Qbo\ApiParameterObjects\QboUpdateOrCreateJournalsApo;
use Modules\Qbo\ApiParameterObjects\QboUpdateOrCreatePaymentsApo;
use Modules\Qbo\ApiParameterObjects\QboUpdateOrCreatePurchaseOrdersApo;
use Modules\Qbo\Entities\QboContact;
use Modules\Qbo\Entities\QboItem;
use Modules\Qbo\Entities\QboVendor;

abstract class AbstractAccountingManager extends AbstractAccountingIntegrationManager
{
    private NominalCodeRepository $nominalCodes;

    private SettingRepository $settings;

    public function __construct(
        protected IntegrationInstanceInterface $integrationInstance,
        protected AccountingAccountRepositoryInterface $accounts,
        protected AccountingContactRepositoryInterface $contacts,
        protected AccountingTaxRateRepositoryInterface $taxRates,
        protected AccountingTaxCodeRepositoryInterface $taxCodes,
        protected AccountingVendorRepositoryInterface $vendors,
        protected AccountingProductRepositoryInterface $products,
        protected AccountingCustomerRepositoryInterface $customers,
        protected AccountingPurchaseOrderRepositoryInterface $purchaseOrders,
        protected AccountingBillRepositoryInterface $bills,
        protected AccountingInvoiceRepositoryInterface $invoices,
        protected AccountingPaymentRepositoryInterface $payments,
        protected AccountingSalesCreditRepositoryInterface $salesCredits,
        protected AccountingJournalRepositoryInterface $journals,
        protected AccountingClientInterface $client
    ) {
        parent::__construct($integrationInstance);
        $this->nominalCodes = app(NominalCodeRepository::class);
        $this->settings = app(SettingRepository::class);
    }

    abstract public function makeTransactionJob(string $type, LazyCollection $transactions);

    abstract public function makeDeletePaymentJob(Collection $transactions);

    abstract public function makePaymentJob(string $type, Collection $transactions);

    // abstract public function handleUpdateOrCreatePurchaseOrdersResponse(ClientResponseDataInterface $response);

    /*
    |--------------------------------------------------------------------------
    | Fetch from API
    |--------------------------------------------------------------------------
    */

    public function fetchAccounts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getAccounts($parameters);
    }

    public function fetchInvoices(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getInvoices($parameters);
    }

    public function fetchPurchaseOrders(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getPurchaseOrders($parameters);
    }

    public function fetchPayments(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getPayments($parameters);
    }

    public function fetchTaxRates(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getTaxRates($parameters);
    }

    public function fetchTaxCodes(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getTaxCodes($parameters);
    }

    public function fetchVendors(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->getVendors($parameters);
    }

    public function createVendor(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->createVendor($parameters);
    }

    public function createProduct(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->createProduct($parameters);
    }

    public function createCustomer(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->client->createCustomer($parameters);
    }

    /*
    |--------------------------------------------------------------------------
    |  to DB
    |--------------------------------------------------------------------------
    */

    public function refreshAccounts(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->refreshData($parameters, [$this, 'fetchAccounts'], [$this->accounts, 'saveForIntegration']);
    }

    public function refreshTaxRates(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->refreshData($parameters, [$this, 'fetchTaxRates'], [$this->taxRates, 'saveForIntegration']);
    }

    public function refreshTaxCodes(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->refreshData($parameters, [$this, 'fetchTaxCodes'], [$this->taxCodes, 'saveForIntegration']);
    }

    public function refreshVendors(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->refreshData($parameters, [$this, 'fetchVendors'], [$this->vendors, 'saveForIntegration']);
    }

    public function refreshInvoices(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        return $this->refreshData($parameters, [$this, 'fetchInvoices'], [$this->invoices, 'saveForIntegration']);
    }

    /*
    |--------------------------------------------------------------------------
    | Other
    |--------------------------------------------------------------------------
    */

    private function saveAndLinkAccountingTransaction(
        AbstractAccountingTransactionRepository|AccountingTransactionRepositoryInterface $repository,
        ApiDataTransformerInterface $parameters,
        string $clientApiCallMethodName
    ) {
        // This is used to link the accounting_transactions
        $uniqueField = $repository->getModelClassName()::getUniqueField();

        // API Call
        $response = $this->client->{$clientApiCallMethodName}($parameters);

        // Save records
        $savedTransactions = $repository->saveAndGetRecordsForIntegration($this->integrationInstance, $response->collection);

        //Un-successful attempts
        //TODO: Handle last error

        // Link account_transactions
        $parameters->accountingTransactions->each(function (AccountingTransaction $accountingTransaction) use ($response, $savedTransactions, $uniqueField, $repository) {
            //TODO: Optimize
            if ($accountingTransaction->type === AccountingTransaction::TYPE_PURCHASE_ORDER) {
                $accountingTransactionNumber = $accountingTransaction->link->purchase_order_number;
            } elseif ($accountingTransaction->type === AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE) {
                $accountingTransactionNumber = $accountingTransaction->link->supplier_invoice_number;
            } elseif ($accountingTransaction->type === AccountingTransaction::TYPE_SALES_ORDER_INVOICE) {
                $accountingTransactionNumber = $accountingTransaction->link->sales_order_number;
            } elseif ($accountingTransaction->type === AccountingTransaction::TYPE_SALES_CREDIT) {
                $accountingTransactionNumber = $accountingTransaction->link->sales_credit_number;
            } elseif (in_array($accountingTransaction->type, AccountingTransaction::TYPES_JOURNAL)) {
                $accountingTransactionNumber = $accountingTransaction->id;
            } else {
                throw new \Exception('Error Processing Request', 1);
            }

            $filteredRecord = $response
                ->collection
                ->filter(function ($record) use ($repository, $accountingTransactionNumber) {
                    if ($record->json_object[$repository->getModelClassName()::getDocNumberUniqueField()] == $accountingTransactionNumber) {
                        return $record;
                    }

                    return null;
                })
                ->first();

            // Successful
            if ($filteredRecord) {
                $repository->linkTransactionWithAccountingTransaction(
                    $accountingTransaction,
                    $savedTransactions->where($uniqueField, $filteredRecord->{$uniqueField})->first()
                );

                return;
            }

            // Error
            $response->errors->filter(function ($record) use ($repository, $accountingTransaction, $accountingTransactionNumber) {
                if (isset($record['response'])) {
                    if ($record['request'][$repository->getModelClassName()::getDocNumberUniqueField()] == $accountingTransactionNumber) {
                        $accountingTransaction->last_sync_error = [formatArrayToString($record['response'])];
                        $accountingTransaction->update();
                    }
                }

                return null;
            })
                ->first();
        });

        return $response;
    }

    private function saveAndLinkPayment(
        AccountingPaymentRepositoryInterface $repository,
        ApiDataTransformerInterface $parameters,
        string $clientApiCallMethodName
    ) {
        // This is used to link the accounting_transactions
        $uniqueField = $repository->getModelClassName()::getUniqueField();

        // API Call
        $response = $this->client->{$clientApiCallMethodName}($parameters);

        // Save records
        $savedTransactions = $repository->saveAndGetRecordsForIntegration($this->integrationInstance, $response->collection);

        //Un-successful attempts
        //TODO: Handle last error

        // Link account_transactions
        $parameters->payments->each(function (Payment $payment) use ($response, $savedTransactions, $uniqueField, $repository) {
            $skuDocNumberUniqueField = null;
            if ($payment->link_type === SalesOrder::class) {
                $skuDocNumberUniqueField = $payment->external_reference;
            } elseif ($payment->link_type === SalesCredit::class) {
                $skuDocNumberUniqueField = $payment->link->salesOrder->payments->first()->external_reference; //TODO: Check with Kalvin
            }

            $filteredRecord = $response->collection->filter(function ($record) use ($repository, $skuDocNumberUniqueField) {
                if ($record->json_object[$repository->getModelClassName()::getDocNumberUniqueField()] == $skuDocNumberUniqueField) {
                    return $record;
                }

                return null;
            })
                ->first();

            if ($filteredRecord) {
                $this->payments->linkPaymentWithAccountingPayment(
                    $payment,
                    $savedTransactions->where($uniqueField, $filteredRecord->{$uniqueField})->first()
                );
            }
        });

        return $response;
    }

    public function createNominalCodes(array $ids)
    {
        $this->nominalCodes->createMany(
            $this->accounts->getAccountsWithoutNominalCodesIn($ids)
                ->map(function (AbstractAccountingAccount $account) {
                    return $account->getNominalCodeData();
                })
                ->toArray()
        );
    }

    public function syncSalesCredits(ApiDataTransformerInterface|QboUpdateOrCreateCreditMemosApo $parameters)
    {
        return $this->saveAndLinkAccountingTransaction($this->salesCredits, $parameters, 'updateOrCreateSalesCredits');
    }

    public function syncInvoicePayments(ApiDataTransformerInterface|QboUpdateOrCreatePaymentsApo $parameters)
    {
        return $this->saveAndLinkPayment($this->payments, $parameters, 'updateOrCreatePayments');
    }

    public function syncPurchaseOrders(ApiDataTransformerInterface|QboUpdateOrCreatePurchaseOrdersApo $parameters)
    {
        return $this->saveAndLinkAccountingTransaction($this->purchaseOrders, $parameters, 'updateOrCreatePurchaseOrders');
    }

    public function syncBills(ApiDataTransformerInterface $parameters)
    {
        return $this->saveAndLinkAccountingTransaction($this->bills, $parameters, 'updateOrCreateBills');
    }

    public function syncInvoices(ApiDataTransformerInterface|QboUpdateOrCreateInvoicesApo $parameters)
    {
        return $this->saveAndLinkAccountingTransaction($this->invoices, $parameters, 'updateOrCreateInvoices');
    }

    public function syncJournals(ApiDataTransformerInterface|QboUpdateOrCreateJournalsApo $parameters)
    {
        return $this->saveAndLinkAccountingTransaction($this->journals, $parameters, 'updateOrCreateJournals');
    }

    public function syncVendors(): void
    {
        $this->vendors->getVendorsNeedingUpdate()->each(function (Supplier $supplier) {
            $vendor = QboVendor::where('DisplayName', $supplier->name)->first();

            if (is_null($vendor)) {
                $vendorResponse = $this->createVendor(new QboCreateVendorApo($supplier->name));

                if ($vendor = $vendorResponse->collection->first()) {
                    $this->vendors->syncVendor($this->integrationInstance, $supplier, $vendor);
                } else {
                    //Try to create with prefix
                    $vendorResponse = $this->createVendor(new QboCreateVendorApo('Vendor-'.$supplier->name));

                    if ($vendor = $vendorResponse->collection->first()) {
                        $this->vendors->syncVendor($this->integrationInstance, $supplier, $vendor);
                    }
                }
            }
        });
    }

    public function syncProducts(array $products): void
    {
        collect($products)->each(function (string $productName) {
            //TODO: Abstract
            $product = QboItem::where('Name', $productName)->first();

            //TODO: Handle
            if ((! isset($product->json_object['IncomeAccountRef']) || ! isset($product->json_object['ExpenseAccountRef'])) && $setting = $this->settings->getSettingByKey(Setting::KEY_NC_MAPPING_COGS)) {
                $incomeChartOfAccount = $this->nominalCodes->getCodeById($setting);
            }

            if (is_null($product) || is_null(@$product->json_object['IncomeAccountRef']) || is_null(@$product->json_object['ExpenseAccountRef'])) {
                $this->products->saveForIntegration(
                    $this->integrationInstance,
                    $this->createProduct(
                        new QboCreateItemApo(
                            $productName,
                            $incomeChartOfAccount,
                            $incomeChartOfAccount,
                            $product->SyncToken ?? null
                        )
                    )->collection
                );
            }
        });
    }

    public function syncCustomers(array $customerNames): void
    {
        collect($customerNames)->each(function (string $customerName) {
            if ($customerName) {
                $customer = QboContact::where('DisplayName', $customerName)->first();

                if (is_null($customer)) {
                    $this->customers->saveForIntegration(
                        $this->integrationInstance,
                        $this->createCustomer(new QboCreateCustomerApo($customerName))->collection
                    );
                }
            }
        });
    }

    public function syncDeletePayments(ApiDataTransformerInterface $parameters): ClientResponseDataInterface
    {
        $response = $this->client->deletePayments($parameters);

        $this->payments->deletePayments($response->collection->toArray());

        return $response;
    }

    /*
    |--------------------------------------------------------------------------
    | Sync Accounting Transactions
    |--------------------------------------------------------------------------
    */

    public function syncTransactions(array $accounting_transaction_ids = [], ?string $modelType = null, ?int $limit = null): void
    {
        switch ($modelType) {
            case SalesOrder::class:
                $this->updateTransactions($this->invoices->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesOrder::class));
                break;

            case PurchaseOrder::class:
                $this->updateTransactions($this->purchaseOrders->getTransactionsNeedingUpdate($accounting_transaction_ids, PurchaseOrder::class));
                break;

            case PurchaseInvoice::class:
                $this->updateTransactions($this->bills->getTransactionsNeedingUpdate($accounting_transaction_ids, PurchaseInvoice::class));
                break;

            case SalesCredit::class:
                $this->updateTransactions($this->salesCredits->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesCredit::class));
                break;

            case StockTake::class:
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, StockTake::class));
                break;

            case InventoryAdjustment::class:
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, InventoryAdjustment::class));
                break;

            case CustomerReturn::class:
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, CustomerReturn::class));
                break;

            case SalesOrderFulfillment::class:
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesOrderFulfillment::class));
                break;

            default:
                $this->updateTransactions($this->invoices->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesOrder::class));
                $this->updateTransactions($this->purchaseOrders->getTransactionsNeedingUpdate($accounting_transaction_ids, PurchaseOrder::class));
                $this->updateTransactions($this->bills->getTransactionsNeedingUpdate($accounting_transaction_ids, PurchaseInvoice::class));
                $this->updateTransactions($this->salesCredits->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesCredit::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, StockTake::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, InventoryAdjustment::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, CustomerReturn::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesOrderFulfillment::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, PurchaseOrderShipmentReceipt::class));
                $this->updateTransactions($this->journals->getTransactionsNeedingUpdate($accounting_transaction_ids, SalesCreditReturn::class));
                break;
        }
    }

    public function syncPayments(array $payment_ids = [], ?string $modelType = null, ?int $limit = null): void
    {
        // Separate calls are because of left join for accounting tables
        switch ($modelType) {
            case SalesOrder::class:
                $this->updatePayments($this->payments->getPaymentsNeedingUpdate($payment_ids, SalesOrder::class), $limit);
                break;

            default:
                $this->updatePayments($this->payments->getPaymentsNeedingUpdate($payment_ids, SalesOrder::class), $limit);
                $this->updatePayments($this->payments->getPaymentsNeedingUpdate($payment_ids, SalesCredit::class), $limit);
                break;
        }
    }

    // this function acts like a router
    public function updateTransactions(LazyCollection $accountingTransactions): void
    {
        collect([
            AccountingTransaction::TYPE_PURCHASE_ORDER_RECEIPT,
            AccountingTransaction::TYPE_CUSTOMER_RETURN,
            AccountingTransaction::TYPE_INVENTORY_ADJUSTMENT,
            AccountingTransaction::TYPE_STOCK_TAKE,
            AccountingTransaction::TYPE_SALES_ORDER_FULFILLMENT,
            AccountingTransaction::TYPE_PURCHASE_ORDER,
            AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE,
            AccountingTransaction::TYPE_SALES_ORDER_INVOICE,
            AccountingTransaction::TYPE_SALES_CREDIT,
        ])
            ->each(function ($accountTransactionType) use ($accountingTransactions) {
                $accountingTransactions
                    ->filter(function (AccountingTransaction $transaction) use ($accountTransactionType) {
                        return $transaction->type == $accountTransactionType;
                    })
                    ->chunk(100)
                    ->each(function (LazyCollection $records) use ($accountTransactionType) {
                        dispatch(
                            $this->makeTransactionJob(
                                $accountTransactionType,
                                $records
                            )
                        )->onQueue('accounting');
                    });
            });
    }

    public function updatePayments(Collection $payments): void
    {
        collect([
            SalesOrder::class,
            SalesCredit::class,
        ])
            ->each(function (string $linkType) use ($payments) {
                $payments
                    ->filter(function (Payment $transaction) use ($linkType) {
                        return $transaction->link_type == $linkType;
                    })
                    ->chunk(100)
                    ->each(function (Collection $records) use ($linkType) {
                        dispatch(
                            $this->makePaymentJob(
                                $linkType,
                                $records
                            )
                        )->onQueue('accounting');
                    });
            });
    }

    public function deletePayments(array $payment_ids = [])
    {
        return dispatch($this->makeDeletePaymentJob($this->payments->getPaymentsNeedingDelete($payment_ids)));
    }

    public function updateOrCreateSuppliers(array $supplier_ids = [])
    {
        $suppliers = $this->contacts->getSuppliersNeedingUpdate($supplier_ids);
        if ($suppliers->count()) {
            //            $parameters =
            //            $this->client->updateOrCreateSuppliers($parameters);
        }
    }
}
