<?php

namespace App\Abstractions;

use Exception;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Spatie\LaravelData\Data;
use Spatie\LaravelData\Optional;

class AbstractRepository
{
    /*
     * An improved upsert function with the following benefits:
     * - Delegates unique field checking to the model
     * - Built in chunking
     * - Returns the unique fields of the upserted data
     */
    public function save(Data|Collection $data, string $modelClass): Collection
    {
        $model = app($modelClass);
        $uniqueFields = $model::getUniqueFields();

        if ($data instanceof Data) {
            $data = Data::collection([$data])->toCollection();
        }

        $data = $data->reject(function ($item) {
            return $item instanceof Optional || ! $item;
        });

        $resultData = collect();

        $data->chunk(1000)->each(function ($chunk) use ($uniqueFields, $model, $resultData, $modelClass) {

            try {
                $model::upsert($chunk->toArray(), $uniqueFields);
            } catch (Exception $e) {
                customlog('upsert_exceptions', $modelClass, [
                    'unique_fields' => $uniqueFields,
                    'data' => $chunk->toArray(),
                    'exception' => $e->getMessage(),
                ]);
                throw $e;
            }

            $selectQuery = $model::query();
            $chunk->each(function (Data $dataChunk) use ($uniqueFields, $selectQuery) {
                $selectQuery->orWhere(function ($query) use ($dataChunk, $uniqueFields) {
                    foreach ($uniqueFields as $uniqueField) {
                        if ($dataChunk->{$uniqueField} instanceof Optional) {
                            $query->whereNull($uniqueField);
                        } else {
                            $query->where($uniqueField, $dataChunk->{$uniqueField});
                        }
                    }
                });
            });
            $resultData->add($selectQuery->select(Arr::prepend($uniqueFields, 'id'))->get()->toArray());
        });

        return $resultData->flatten(1);
    }

    public function getForValues(array $values, string $fieldName, string $modelClass, array $eagerLoads = [], $chunkSize = 10000, $limit = null, $offset = null): EloquentCollection
    {
        $records = new EloquentCollection();

        foreach (array_chunk($values, $chunkSize) as $chunk) {
            $chunkRecords = app($modelClass)::with($eagerLoads)->whereIn($fieldName, $chunk);

            if ($limit) {
                $chunkRecords->limit($limit);
            }

            if ($offset) {
                $chunkRecords->offset($offset);
            }

            $records = $records->merge($chunkRecords->get());
        }

        return $records;
    }

    public function getForValue(string $value, string $fieldName, string $modelClass): ?Model
    {
        return app($modelClass)::where($fieldName, $value)->first();
    }

    public function setValueForIds(array $ids, string $modelClass, array $values): void
    {
        app($modelClass)::whereIn('id', $ids)->update($values);
    }
}
