<?php

namespace Modules\Xero\Tests\Feature;

use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Services\Accounting\AccountingTransactionManager;
use Exception;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Mockery\MockInterface;
use Modules\Amazon\Entities\AmazonIntegrationInstance;
use Modules\Xero\Collections\XeroCreditNoteCollection;
use Modules\Xero\Collections\XeroInvoiceCollection;
use Modules\Xero\DTO\XeroCreditNote;
use Modules\Xero\DTO\XeroInvoice;
use Modules\Xero\Repositories\XeroTransactionRepository;
use Modules\Xero\Services\Client;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;

class XeroTransactionRepositoryTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();

        $integration = Integration::query()->whereName(Integration::NAME_XERO)->first();

        // Create the Xero integration
        $integrationInstance = IntegrationInstance::factory()->create([
            'name' => Integration::NAME_XERO,
            'connection_settings' => [
                'clientId' => $this->faker->md5(),
                'clientSecret' => $this->faker->numberBetween(1, 10),
            ],
            'integration_id' => $integration ? $integration->id : Integration::factory()->create([
                'name' => Integration::NAME_XERO,
                'integration_type' => Integration::TYPE_ACCOUNTING,
            ]),
        ]);

        $integrationInstance->update([
            'integration_settings' => [
                'settings' => [
                    'sync_start_date' => now()->subYears(2)->toDateTimeString(),
                ],
            ],
        ]);

        // Ensure that all external api requests are mocked.
        Http::preventStrayRequests();

        $this->mockConnectionRequests();
    }

    protected function mockConnectionRequests()
    {
        Http::fake([
            'identity.xero.com/connect/token' => Http::response([
                'access_token' => $this->faker->sha1(),
                'refresh_token' => $this->faker->sha1(),
                'id_token' => $this->faker->sha1(),
                'expires_in' => now()->addDays(5)->unix(),
            ]),
            'api.xero.com/connections' => Http::response([
                ['tenantId' => $this->faker->sha1()],
            ]),
        ]);
    }

    protected function mockEndpoint(string $endpoint, array $response = [], array $headers = [])
    {
        $endpoint = ltrim($endpoint, '/');
        Http::fake([
            "api.xero.com/api.xro/2.0/$endpoint" => Http::response(
                body: json_encode($response),
                headers: $headers + [
                    'X-DayLimit-Remaining' => 100,
                ]
            ),
        ]);
    }

    public function test_it_can_handle_validation_errors_for_invoice(): void
    {
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->hasSalesOrderLines()->create();

        $reference = $this->faker->word();

        /** @var AccountingTransaction $transaction */
        $transaction = AccountingTransaction::factory()
            ->has(AccountingTransactionLine::factory(
                state: [
                    'type' => AccountingTransactionLine::TYPE_SALES_INVOICE_LINE,
                    'quantity' => 3,
                    'link_id' => $order->salesOrderLines->first()->id,
                    'link_type' => SalesCreditLine::class,
                ]
            ))
            ->create([
                'type' => AccountingTransaction::TYPE_SALES_ORDER_INVOICE,
                'reference' => $reference,
                'link_type' => SalesOrder::class,
                'link_id' => $order->id,
            ]);

        $invoice = XeroInvoice::from(
            [
                'InvoiceID' => $transaction->id,
                'InvoiceNumber' => $transaction->reference,
                'Type' => XeroCreditNote::TYPE_ACCPAYCREDIT,
                'LineItems' => [
                    [
                        'Description' => $this->faker->sentence(),
                        'UnitAmount' => $this->faker->numberBetween(10, 30),
                        'TaxAmount' => $this->faker->numberBetween(0, 10),
                        'Quantity' => 3,
                    ],
                ],
            ]
        );

        $this->mockEndpoint(
            endpoint: '/Invoices?*',
            response: [
                'Invoices' => [
                    $invoice,
                ],
            ]
        );

        // Keeping this mock so the test works in the branch. May not be needed later on.
        $this->mock(Client::class, function (MockInterface $mock) use ($invoice) {
            $invoices = new XeroInvoiceCollection();

            $invoices->add(
                $invoice
            );

            $mock->shouldReceive('updateOrCreateInvoices')->andReturn($invoices);
        });

        $repository = new XeroTransactionRepository();
        $repository->updateXeroTransactions(new Collection([$transaction]));

        $this->assertDatabaseMissing('xero_transactions', [
            'last_error' => null,
        ]);
    }

    public function test_it_can_handle_validation_errors_for_credit_notes(): void
    {
        /** @var SalesOrder $order */
        $order = SalesOrder::factory()->hasSalesOrderLines()->create();

        /** @var SalesCredit $credit */
        $credit = SalesCredit::factory()->hasSalesCreditLines()->create([
            'sales_order_id' => $order->id,
        ]);

        $reference = $this->faker->word();

        /** @var AccountingTransaction $transaction */
        $orderTransaction = AccountingTransaction::factory()
            ->has(AccountingTransactionLine::factory(
                state: [
                    'type' => AccountingTransactionLine::TYPE_SALES_INVOICE_LINE,
                    'quantity' => 3,
                    'link_id' => $order->salesOrderLines->first()->id,
                    'link_type' => SalesOrderLine::class,
                ]
            ))
            ->create([
                'type' => AccountingTransaction::TYPE_SALES_ORDER_INVOICE,
                'reference' => $reference,
                'link_type' => SalesOrder::class,
                'link_id' => $order->id,
            ]);

        /** @var AccountingTransaction $transaction */
        $transaction = AccountingTransaction::factory()
            ->has(AccountingTransactionLine::factory(
                state: [
                    'type' => AccountingTransactionLine::TYPE_SALES_CREDIT_LINE,
                    'quantity' => 3,
                    'link_id' => $credit->salesCreditLines->first()->id,
                    'link_type' => SalesCreditLine::class,
                ]
            ))
            ->create([
                'type' => AccountingTransaction::TYPE_SALES_CREDIT,
                'reference' => $reference,
                'link_type' => SalesCredit::class,
                'link_id' => $credit->id,
                'parent_id' => $orderTransaction->id,
            ]);

        $creditNote = XeroCreditNote::from(
            [
                'CreditNoteID' => $transaction->id,
                'CreditNoteNumber' => $transaction->reference,
                'Type' => XeroCreditNote::TYPE_ACCPAYCREDIT,
                'LineItems' => [
                    [
                        'Description' => $this->faker->sentence(),
                        'UnitAmount' => $this->faker->numberBetween(10, 30),
                        'TaxAmount' => $this->faker->numberBetween(0, 10),
                        'Quantity' => 3.00004,
                    ],
                ],
            ]
        );

        $this->mockEndpoint(
            endpoint: '/CreditNotes?*',
            response: [
                'CreditNotes' => [
                    $creditNote,
                ],
            ]
        );

        // Keeping this mock so the test works in the branch. Should be removed when the xero test is merged.
        $this->mock(Client::class, function (MockInterface $mock) use ($creditNote) {
            $creditNotes = new XeroCreditNoteCollection;

            $creditNotes->add(
                $creditNote
            );

            $mock->shouldReceive('updateOrCreateCreditNotes')->andReturn($creditNotes);
        });

        $repository = new XeroTransactionRepository();
        $repository->updateXeroTransactions(new Collection([$transaction]));

        $this->assertDatabaseMissing('xero_transactions', [
            'last_error' => null,
        ]);
    }

    /**
     * @throws Exception
     */
    public function test_it_wont_sync_batchable_transactions(): void
    {
        Queue::fake();

        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'integration_settings' => [
                'batch_sales_order_invoices' => true,
            ],
        ])->hasSalesChannel()->create();

        $salesChannel = $amazonIntegrationInstance->salesChannel;

        SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
        ])->create();

        app(AccountingTransactionManager::class)->sync();

        $this->assertEquals(0, app(XeroTransactionRepository::class)->getTransactionsNeedingUpdate()->count());
    }

    public function test_it_sync_enabled_sales_order_accounting_transactions(): void
    {
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'integration_settings' => [
                'sync_sales_order_invoices_to_accounting' => true,
            ],
        ])->hasSalesChannel()->create();

        $salesChannel = $amazonIntegrationInstance->salesChannel;

        SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
        ])->create();

        app(AccountingTransactionManager::class)->sync();

        $this->assertEquals(1, app(XeroTransactionRepository::class)->getTransactionsNeedingUpdate()->count());
    }

    public function test_it_wont_sync_not_enabled_sales_order_accounting_transactions(): void
    {
        $amazonIntegrationInstance = AmazonIntegrationInstance::factory([
            'integration_settings' => [
                'sync_sales_order_invoices_to_accounting' => false,
            ],
        ])->hasSalesChannel()->create();

        $salesChannel = $amazonIntegrationInstance->salesChannel;

        SalesOrder::factory([
            'sales_channel_id' => $salesChannel->id,
        ])->create();
        app(AccountingTransactionManager::class)->sync();

        $this->assertEquals(0, app(XeroTransactionRepository::class)->getTransactionsNeedingUpdate()->count());
    }
}
