<?php

namespace Modules\Xero\Repositories;

use App\Abstractions\AbstractRepository;
use App\Exceptions\ApiException;
use App\Helpers;
use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\Setting;
use App\Repositories\Accounting\AccountingTransactionRepository;
use App\Repositories\IntegrationInstanceRepository;
use App\Repositories\NominalCodeRepository;
use Carbon\Carbon;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Collection as EloquentCollection;
use Illuminate\Database\Query\JoinClause;
use Illuminate\Support\Arr;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use InvalidArgumentException;
use Modules\Xero\Collections\XeroManualJournalCollection;
use Modules\Xero\DTO\XeroContact as XeroContactDTO;
use Modules\Xero\DTO\XeroCreditNote;
use Modules\Xero\DTO\XeroCreditNoteAllocation;
use Modules\Xero\DTO\XeroCreditNotes;
use Modules\Xero\DTO\XeroInvoice;
use Modules\Xero\DTO\XeroInvoices;
use Modules\Xero\DTO\XeroJournalLine;
use Modules\Xero\DTO\XeroLineItems;
use Modules\Xero\DTO\XeroManualJournalDto;
use Modules\Xero\DTO\XeroManualJournals;
use Modules\Xero\DTO\XeroPurchaseOrder;
use Modules\Xero\DTO\XeroPurchaseOrders;
use Modules\Xero\Entities\XeroManualJournal;
use Modules\Xero\Entities\XeroTransaction;
use Modules\Xero\Enums\XeroLockedTransactionErrorEnum;
use Modules\Xero\Jobs\XeroUpdateOrCreateCreditNotesJob;
use Modules\Xero\Jobs\XeroUpdateOrCreateInvoicesJob;
use Modules\Xero\Jobs\XeroUpdateOrCreateManualJournalsJob;
use Modules\Xero\Jobs\XeroUpdateOrCreatePurchaseOrdersJob;
use Modules\Xero\Jobs\XeroVoidCreditNotesJob;
use Modules\Xero\Jobs\XeroVoidInvoicesJob;
use Modules\Xero\Jobs\XeroVoidManualJournalsJob;
use Modules\Xero\Jobs\XeroVoidPurchaseOrdersJob;
use Modules\Xero\Services\Client;
use Spatie\LaravelData\Data;
use Spatie\LaravelData\Optional;
use Throwable;

class XeroTransactionRepository extends AbstractRepository
{
    protected XeroContactRepository $xeroContactRepository;

    protected XeroTaxRateRepository $xeroTaxRateRepository;

    protected NominalCodeRepository $nominalCodeRepository;

    protected AccountingTransactionRepository $accountingTransactionRepository;

    protected int $batchSizeManualJournal = 100;

    public function __construct()
    {
        $this->xeroContactRepository = app(XeroContactRepository::class);
        $this->xeroTaxRateRepository = app(XeroTaxRateRepository::class);
        $this->nominalCodeRepository = app(NominalCodeRepository::class);
        $this->accountingTransactionRepository = app(AccountingTransactionRepository::class);
    }

    public function getTransactionsNeedingUpdate(array $accounting_transaction_ids = []): EloquentCollection
    {
        $query = AccountingTransaction::query()
            ->leftJoin('xero_transactions', function (JoinClause $join) {
                $join->on('xero_transactions.id', 'accounting_transactions.accounting_integration_id');
                $join->where('accounting_transactions.accounting_integration_type', XeroTransaction::class);
            })
            ->where('is_sync_enabled', true)
            ->whereNull('submission_started_at')
            ->where('is_batchable', false)
            ->where('transaction_date', '>=', app(IntegrationInstanceRepository::class)->getAccountingInstance()->integration_settings['settings']['sync_start_date']);
        // See SKU-5496 for why this is being disabled
        /*->where(function (Builder $query) {
            $query->whereNull('xero_transactions.id');
            $query->orWhere('status', '!=', XeroInvoice::STATUS_PAID);
            $query->orWhereNull('status');
        });*/

        if ($accounting_transaction_ids) {
            $query->whereIn('accounting_transactions.id', $accounting_transaction_ids);
        } else {
            $query->where(function (Builder $query) {
                $query->whereNull('accounting_integration_id');
                $query->orWhereColumn('accounting_transactions.updated_at', '>', 'xero_transactions.updated_at');
                $query->orWhereNull('xero_transactions.updated_at');
            });
        }

        Log::channel('xero')->info('Syncing '.$query->count().' Xero transactions');

        return $query->get(['accounting_transactions.*']);
    }

    public function getXeroUuidsFromAccountingTransactionIds(array $accounting_transaction_ids = []): Collection
    {
        return AccountingTransaction::query()
            ->select('accounting_transactions.type', 'xero_transactions.xero_uuid')
            ->join('xero_transactions', function (JoinClause $join) {
                $join->on('xero_transactions.id', 'accounting_transactions.accounting_integration_id');
                $join->where('accounting_transactions.accounting_integration_type', XeroTransaction::class);
            })
            ->whereIn('accounting_transactions.id', $accounting_transaction_ids)
            ->get();
    }

    public function getXeroUuidsForErroredTransactions(): Collection
    {
        return XeroTransaction::query()
            ->select('accounting_transactions.type', 'xero_transactions.xero_uuid')
            ->join('accounting_transactions', function (JoinClause $join) {
                $join->on('accounting_transactions.accounting_integration_id', 'xero_transactions.id');
                $join->where('accounting_transactions.accounting_integration_type', XeroTransaction::class);
                $join->whereIn('accounting_transactions.type', [AccountingTransaction::TYPE_SALES_CREDIT, AccountingTransaction::TYPE_SALES_ORDER_INVOICE]);
            })
            ->whereNotNull('xero_uuid')
            ->whereNotNull('last_error')
            ->get();
    }

    public function getTransactionsNeedingVoid(array $accounting_transaction_ids = []): EloquentCollection
    {
        $query = XeroTransaction::query()
            ->leftJoin('accounting_transactions', function (JoinClause $join) {
                $join->on('xero_transactions.id', 'accounting_transactions.accounting_integration_id');
                $join->where('accounting_transactions.accounting_integration_type', XeroTransaction::class);
            })
            ->whereNull('accounting_integration_id')
            ->whereNotNull('xero_uuid');

        if ($accounting_transaction_ids) {
            $query->whereIn('accounting_transactions.id', $accounting_transaction_ids);
        }

        return $query->get(['xero_transactions.*']);
    }

    public function getTransactionsNeedingAllocation(array $accounting_transaction_ids = []): EloquentCollection
    {
        $query = AccountingTransaction::query()
            ->whereHas('parent')
            ->whereHas('accountingIntegration')
            ->where('type', AccountingTransaction::TYPE_SALES_CREDIT);

        if ($accounting_transaction_ids) {
            $query->whereIn('accounting_transactions.id', $accounting_transaction_ids);
        } else {
            $query->where(function (Builder $query) {
                $query->whereNull('allocated_at');
                $query->orWhereColumn('updated_at', '>', 'allocated_at');
            });
        }

        return $query->get();
    }

    public function purgeDeletedTransactions(): void
    {
        XeroTransaction::query()->where('status', 'DELETED')->delete();
    }

    /**
     * @throws Exception
     */
    public function voidXeroTransactions(Collection $transactions): void
    {
        $transactions->filter(function (XeroTransaction $transaction) {
            return in_array($transaction->type, XeroTransaction::TYPES_INVOICE);
        })->chunk(100)->each(function (Collection $invoices) {
            dispatch(new XeroVoidInvoicesJob($invoices))->onQueue('xero');
        });

        $transactions->filter(function (XeroTransaction $transaction) {
            return $transaction->type == XeroTransaction::TYPE_PURCHASE_ORDER;
        })->chunk(100)->each(function (Collection $purchaseOrders) {
            dispatch(new XeroVoidPurchaseOrdersJob($purchaseOrders))->onQueue('xero');
        });

        $transactions->filter(function (XeroTransaction $transaction) {
            return $transaction->type == XeroTransaction::TYPE_ACCRECCREDIT;
        })->chunk(100)->each(function (Collection $creditNotes) {
            dispatch(new XeroVoidCreditNotesJob($creditNotes))->onQueue('xero');
        });

        $transactions->filter(function (XeroTransaction $transaction) {
            return $transaction->type == XeroTransaction::TYPE_JOURNAL;
        })->chunk(100)->each(function (Collection $journals) {
            dispatch(new XeroVoidManualJournalsJob($journals))->onQueue('xero');
        });
    }

    /**
     * @throws Exception
     */
    public function voidInvoices(Collection $transactions)
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroInvoices = [];
        $transactions->each(function (XeroTransaction $transaction) use (&$xeroInvoices) {

            $xeroInvoices[] = XeroInvoice::from([
                'InvoiceID' => $transaction->xero_uuid,
                'Status' => XeroInvoice::STATUS_VOIDED,
            ]);
        });

        $requestInvoices = XeroInvoices::from([
            'Invoices' => $xeroInvoices,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseInvoices = $client->updateOrCreateInvoices($requestInvoices->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->each(function (XeroTransaction $xeroTransaction) use ($e) {
                $xeroTransaction->last_error = Arr::wrap('Connection error: '.$e->getMessage());
                $xeroTransaction->save();
            });

            return;
        }

        /** @var XeroInvoice $invoice */
        foreach ($responseInvoices as $invoice) {
            DB::transaction(function () use ($invoice) {

                $has_error = ($invoice->StatusAttributeString && $invoice->StatusAttributeString != 'OK') || ! empty((array) $invoice->ValidationErrors);

                /** @var XeroTransaction $xeroTransaction */
                $xeroTransaction = XeroTransaction::query()
                    ->where('xero_uuid', $invoice->InvoiceID)->firstOrFail();

                if ($has_error) {
                    if ($invoice->json_object['Status'] == XeroInvoice::STATUS_VOIDED) {
                        $xeroTransaction->delete();
                    } else {
                        $xeroTransaction->last_error = array_unique(Arr::flatten($invoice->ValidationErrors));
                        $xeroTransaction->json_object = $invoice->json_object;

                        $xeroTransaction->save();
                    }
                } else {
                    $xeroTransaction->delete();
                }
            });
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function voidPurchaseOrders(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroPurchaseOrders = [];
        $transactions->each(function (XeroTransaction $transaction) use (&$xeroPurchaseOrders) {

            $xeroPurchaseOrders[] = XeroPurchaseOrder::from([
                'PurchaseOrderID' => $transaction->xero_uuid,
                'Status' => XeroPurchaseOrder::STATUS_DELETED,
            ]);
        });

        $requestPurchaseOrders = XeroPurchaseOrders::from([
            'PurchaseOrders' => $xeroPurchaseOrders,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responsePurchaseOrders = $client->updateOrCreatePurchaseOrders($requestPurchaseOrders->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->each(function (XeroTransaction $xeroTransaction) use ($e) {
                $xeroTransaction->last_error = Arr::wrap('Connection error: '.$e->getMessage());
                $xeroTransaction->save();
            });

            return;
        }

        /** @var XeroPurchaseOrder $purchaseOrder */
        foreach ($responsePurchaseOrders as $purchaseOrder) {
            DB::transaction(function () use ($purchaseOrder) {

                $has_error = ($purchaseOrder->StatusAttributeString && $purchaseOrder->StatusAttributeString != 'OK') || ! empty((array) $purchaseOrder->ValidationErrors);

                /** @var XeroTransaction $xeroTransaction */
                $xeroTransaction = XeroTransaction::query()
                    ->where('xero_uuid', $purchaseOrder->PurchaseOrderID)->firstOrFail();

                if ($has_error) {
                    if ($purchaseOrder->json_object['Status'] == XeroPurchaseOrder::STATUS_DELETED) {
                        $xeroTransaction->delete();
                    } else {
                        $xeroTransaction->last_error = array_unique(Arr::flatten($purchaseOrder->ValidationErrors));
                        $xeroTransaction->json_object = $purchaseOrder->json_object;
                        $xeroTransaction->save();
                    }
                } else {
                    $xeroTransaction->delete();
                }
            });
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function voidCreditNotes(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroCreditNotes = [];
        $transactions->each(function (XeroTransaction $transaction) use (&$xeroCreditNotes) {

            $xeroCreditNotes[] = XeroCreditNote::from([
                'CreditNoteID' => $transaction->xero_uuid,
                'Status' => XeroCreditNote::STATUS_VOIDED,
            ]);
        });

        $requestCreditNotes = XeroCreditNotes::from([
            'CreditNotes' => $xeroCreditNotes,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseCreditNotes = $client->updateOrCreateCreditNotes($requestCreditNotes->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->each(function (XeroTransaction $xeroTransaction) use ($e) {
                $xeroTransaction->last_error = Arr::wrap('Connection error: '.$e->getMessage());
                $xeroTransaction->save();
            });

            return;
        }

        /** @var XeroCreditNote $creditNote */
        foreach ($responseCreditNotes as $creditNote) {
            DB::transaction(function () use ($creditNote) {

                $has_error = ($creditNote->StatusAttributeString && $creditNote->StatusAttributeString != 'OK') || ! empty((array) $creditNote->ValidationErrors);
                /** @var XeroTransaction $xeroTransaction */
                $xeroTransaction = XeroTransaction::query()
                    ->where('xero_uuid', $creditNote->CreditNoteID)->firstOrFail();

                if ($has_error) {
                    if ($creditNote->json_object['Status'] == XeroCreditNote::STATUS_VOIDED) {
                        $xeroTransaction->delete();
                    } else {
                        $xeroTransaction->last_error = array_unique(Arr::flatten($creditNote->ValidationErrors));
                        $xeroTransaction->json_object = $creditNote->json_object;

                        $xeroTransaction->save();
                    }
                } else {
                    $xeroTransaction->delete();
                }
            });
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function voidManualJournals(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroManualJournals = [];
        $transactions->each(function (XeroTransaction $transaction) use (&$xeroManualJournals) {

            $xeroManualJournals[] = XeroManualJournalDto::from([
                'ManualJournalID' => $transaction->xero_uuid,
                'Status' => XeroCreditNote::STATUS_VOIDED,
            ]);
        });

        $requestManualJournals = XeroManualJournals::from([
            'ManualJournals' => $xeroManualJournals,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseManualJournals = $client->updateOrCreateManualJournals($requestManualJournals->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->each(function (XeroTransaction $xeroTransaction) use ($e) {
                $xeroTransaction->last_error = Arr::wrap('Connection error: '.$e->getMessage());
                $xeroTransaction->save();
            });

            return;
        }

        /** @var XeroManualJournalDto $manualJournal */
        foreach ($responseManualJournals as $manualJournal) {
            DB::transaction(function () use ($manualJournal) {

                $has_error = ($manualJournal->StatusAttributeString && $manualJournal->StatusAttributeString != 'OK') || ! empty((array) $manualJournal->ValidationErrors);

                /** @var XeroTransaction $xeroTransaction */
                $xeroTransaction = XeroTransaction::query()
                    ->where('xero_uuid', $manualJournal->ManualJournalID)->firstOrFail();

                if ($has_error) {
                    if ($manualJournal->json_object['Status'] == XeroCreditNote::STATUS_VOIDED) {
                        $xeroTransaction->delete();
                    } else {
                        $xeroTransaction->last_error = array_unique(Arr::flatten($manualJournal->ValidationErrors));
                        $xeroTransaction->json_object = $manualJournal->json_object;
                        $xeroTransaction->save();
                    }
                } else {
                    $xeroTransaction->delete();
                }
            });
        }
    }

    /**
     * @throws Exception
     * @throws ApiException
     * @throws GuzzleException
     */
    public function getTransactions(Collection $transactions)
    {
        $this->getInvoices($transactions->filter(function ($transaction) {
            return in_array($transaction->type, AccountingTransaction::TYPES_INVOICE);
        }));

        $this->getCreditNotes($transactions->filter(function ($transaction) {
            return $transaction->type == AccountingTransaction::TYPE_SALES_CREDIT;
        }));
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     * @throws Exception
     */
    public function getInvoices(Collection $transactions): void
    {
        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        $responseInvoices = $client->getInvoices($transactions->pluck('xero_uuid')->take(40)->toArray());

        /** @var XeroInvoice $invoice */
        foreach ($responseInvoices as $invoice) {
            /** @var XeroTransaction $xeroTransaction */
            $xeroTransaction = XeroTransaction::query()->firstOrNew([
                'xero_uuid' => $invoice->InvoiceID]);

            $xeroTransaction->last_error = null;
            $xeroTransaction->json_object = $invoice->json_object;

            $xeroTransaction->save();
        }
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     * @throws Exception
     */
    public function getCreditNotes(Collection $transactions): void
    {
        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        // TODO:: Kalvin, should we fetch these in bulk as well?
        $transactions->pluck('xero_uuid')->take(10)->each(function (string $xero_uuid) use ($client) {
            $responseCreditNotes = $client->getCreditNote($xero_uuid);

            /** @var XeroCreditNote $creditNote */
            foreach ($responseCreditNotes as $creditNote) {
                /** @var XeroTransaction $xeroTransaction */
                $xeroTransaction = XeroTransaction::query()->firstOrNew([
                    'xero_uuid' => $creditNote->CreditNoteID]);

                $xeroTransaction->last_error = null;
                $xeroTransaction->json_object = $creditNote->json_object;

                $xeroTransaction->save();
            }
        });
    }

    /**
     * @throws Exception
     */
    public function updateXeroTransactions(Collection $transactions): void
    {
        $transactions->filter(function (AccountingTransaction $transaction) {
            return in_array($transaction->type, AccountingTransaction::TYPES_INVOICE);
        })->chunk(100)->each(function (Collection $invoices) {
            dispatch(new XeroUpdateOrCreateInvoicesJob($invoices))->onQueue('xero');
        });

        $transactions->filter(function (AccountingTransaction $transaction) {
            return $transaction->type == AccountingTransaction::TYPE_PURCHASE_ORDER;
        })->chunk(100)->each(function (Collection $purchaseOrders) {
            dispatch(new XeroUpdateOrCreatePurchaseOrdersJob($purchaseOrders))->onQueue('xero');
        });

        $transactions->filter(function (AccountingTransaction $transaction) {
            return $transaction->type == AccountingTransaction::TYPE_SALES_CREDIT;
        })->chunk(100)->each(function (Collection $creditNotes) {
            dispatch(new XeroUpdateOrCreateCreditNotesJob($creditNotes))->onQueue('xero');
        });

        $transactions->filter(function (AccountingTransaction $transaction) {
            return in_array($transaction->type, AccountingTransaction::TYPES_JOURNAL);
        })->chunk(100)->each(function (Collection $manualJournals) {
            dispatch(new XeroUpdateOrCreateManualJournalsJob($manualJournals))->onQueue('xero');
        });
    }

    /**
     * @throws Throwable
     */
    public function handleXeroTransactionResponse(
        $xeroTransactionResponse,
        $transactionType,
        $uuid,
        $reference,
    ): void
    {
        DB::transaction(function () use ($xeroTransactionResponse, $transactionType, $uuid, $reference) {
            $statusAttributeString = $xeroTransactionResponse->StatusAttributeString ?? null;
            $validationErrors = (array) $xeroTransactionResponse->ValidationErrors ?? [];
            $validationErrors = array_unique(Arr::flatten($validationErrors));

            $has_error = ($statusAttributeString && $statusAttributeString != 'OK') || ! empty($validationErrors);

            /** @var AccountingTransaction $accountingTransaction */
            $accountingTransaction = AccountingTransaction::query()
                ->where('reference', $reference)
                ->firstOrFail();

            if ($uuid == '00000000-0000-0000-0000-000000000000') {
                $accountingTransaction->saveLastSyncError($validationErrors);

                return;
            }

            $xeroTransaction = $accountingTransaction->accountingIntegration ?? new XeroTransaction();

            $xeroTransaction->xero_uuid = $uuid;
            $xeroTransaction->json_object = $xeroTransactionResponse->json_object;
            $xeroTransaction->type = $transactionType;
            $xeroTransaction->last_error = $has_error ? $validationErrors : null;
            $xeroTransaction->timestamps = !$has_error;
            $xeroTransaction->save();

            if (! $accountingTransaction->accountingIntegration) {
                $accountingTransaction->accounting_integration_id = $xeroTransaction->id;
                $accountingTransaction->accounting_integration_type = XeroTransaction::class;
            }
            $accountingTransaction->timestamps = false;
            Log::channel('xero')->debug('Unlocking submission started for '.$accountingTransaction->id);
            $accountingTransaction->submission_started_at = null;
            $accountingTransaction->last_sync_error = $has_error ? $validationErrors : null;
            $accountingTransaction->save();
        });

    }

    /**
     * @throws Throwable
     */
    public function updateOrCreateInvoices(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroInvoices = [];
        $transactions->take(100)->each(function (AccountingTransaction $transaction) use (&$xeroInvoices) {
            $lineItems = [];

            $passOk = true;
            $transaction->accountingTransactionLines()->each(
                function (AccountingTransactionLine $accountingTransactionLine) use (&$lineItems, $transaction, &$xeroTotal, &$skuTotal, &$passOk) {
                    if (! $taxLine = $this->getTaxForAccountingTransactionLine($accountingTransactionLine)) {
                        $passOk = false;

                        return;
                    }
                    $item = [
                        'Description' => $accountingTransactionLine->description,
                        'Quantity' => $accountingTransactionLine->quantity,
                        'UnitAmount' => $accountingTransactionLine->amount +
                            ($transaction->is_tax_included && $accountingTransactionLine->quantity != 0 ?
                                ($accountingTransactionLine->tax_amount / $accountingTransactionLine->quantity) : 0),
                        'AccountCode' => $accountingTransactionLine->nominalCode->code,
                    ] + $taxLine;
                    $lineItems[] = XeroLineItems::from($item);
                }
            );
            if (! $passOk) {
                return;
            }

            $this->handleRounding($lineItems);

            $lineAmountTypes = match ($transaction->is_tax_included) {
                1 => XeroInvoice::LINE_AMOUNT_TYPE_INCLUSIVE,
                0 => XeroInvoice::LINE_AMOUNT_TYPE_EXCLUSIVE,
                default => XeroInvoice::LINE_AMOUNT_TYPE_NO_TAX,
            };

            $xeroInvoice = XeroInvoice::from([
                'Type' => XeroTransaction::TYPES_INVOICE[$transaction->type],
                'Date' => Helpers::dateUtcToLocal($transaction->transaction_date)->format('Y-m-d'),
                'DueDate' => Helpers::dateUtcToLocal($transaction->transaction_date)->format('Y-m-d'),
                'Status' => XeroInvoice::STATUS_AUTHORISED,
                'Url' => $this->getUrl($transaction->id),
                'LineAmountTypes' => $lineAmountTypes,
                'LineItems' => $lineItems,
                'InvoiceNumber' => $transaction->reference,
                'CurrencyCode' => $transaction->currency_code,
                'CurrencyRate' => $transaction->currency_rate == 0 ? 0 : (1 / $transaction->currency_rate),
            ]);

            if ($transaction->name) {
                $xeroInvoice->Contact = XeroContactDTO::from([
                    'Name' => $transaction->name,
                ]);
            }

            $xeroInvoices[] = $xeroInvoice;
        });

        if (empty($xeroInvoices)) {
            return;
        }

        $requestInvoices = XeroInvoices::from([
            'Invoices' => $xeroInvoices,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseInvoices = $client->updateOrCreateInvoices($requestInvoices->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->take(100)->each(function (AccountingTransaction $accountingTransaction) use ($e) {
                $accountingTransaction->saveLastSyncError(Arr::wrap('Connection error: '.$e->getMessage()));
            });

            return;
        }

        /** @var XeroInvoice $invoice */
        foreach ($responseInvoices as $invoice) {
            $this->handleXeroTransactionResponse(
                $invoice,
                $invoice->Type,
                $invoice->InvoiceID,
                $invoice->InvoiceNumber,
            );
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function updateOrCreatePurchaseOrders(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroPurchaseOrders = [];
        $transactions->take(100)->each(function (AccountingTransaction $transaction) use (&$xeroPurchaseOrders) {

            $ContactID = $this->xeroContactRepository->getContactIDFromSupplier($transaction->link->supplier);

            $xeroContactDTO = XeroContactDTO::from([
                'ContactID' => $ContactID,
            ]);

            $lineItems = [];

            $passOk = true;
            $transaction->accountingTransactionLines()->each(
                function (AccountingTransactionLine $accountingTransactionLine) use (&$lineItems, $transaction, &$passOk) {
                    if (! $taxLine = $this->getTaxForAccountingTransactionLine($accountingTransactionLine)) {
                        $passOk = false;

                        return;
                    }
                    $lineItems[] = XeroLineItems::from([
                        'Description' => $accountingTransactionLine->description,
                        'Quantity' => $accountingTransactionLine->quantity,
                        'UnitAmount' => $accountingTransactionLine->amount +
                        ($transaction->is_tax_included && $accountingTransactionLine->quantity != 0 ?
                            ($accountingTransactionLine->tax_amount / $accountingTransactionLine->quantity) : 0),
                        'AccountCode' => $accountingTransactionLine->nominalCode->code,
                    ] + $taxLine);
                }
            );
            if (! $passOk) {
                return;
            }

            $this->handleRounding($lineItems);

            $lineAmountTypes = match ($transaction->is_tax_included) {
                1 => XeroInvoice::LINE_AMOUNT_TYPE_INCLUSIVE,
                0 => XeroInvoice::LINE_AMOUNT_TYPE_EXCLUSIVE,
                default => XeroInvoice::LINE_AMOUNT_TYPE_NO_TAX,
            };

            $xeroPurchaseOrder = XeroPurchaseOrder::from([
                'Contact' => $xeroContactDTO,
                'Date' => Helpers::dateUtcToLocal($transaction->transaction_date)->format('Y-m-d'),
                'Status' => $transaction->link->fully_invoiced ? XeroPurchaseOrder::STATUS_BILLED : XeroPurchaseOrder::STATUS_AUTHORISED,
                'LineAmountTypes' => $lineAmountTypes,
                'LineItems' => $lineItems,
                'CurrencyCode' => $transaction->currency_code,
                'CurrencyRate' => $transaction->currency_rate == 0 ? 0 : (1 / $transaction->currency_rate),
                'PurchaseOrderNumber' => $transaction->reference,
            ]);

            $xeroPurchaseOrders[] = $xeroPurchaseOrder;
        });

        if (empty($xeroPurchaseOrders)) {
            return;
        }

        $requestPurchaseOrders = XeroPurchaseOrders::from([
            'PurchaseOrders' => $xeroPurchaseOrders,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responsePurchaseOrders = $client->updateOrCreatePurchaseOrders($requestPurchaseOrders->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->take(100)->each(function (AccountingTransaction $accountingTransaction) use ($e) {
                $accountingTransaction->saveLastSyncError(Arr::wrap('Connection error: '.$e->getMessage()));
            });

            return;
        }

        /** @var XeroPurchaseOrder $purchaseOrder */
        foreach ($responsePurchaseOrders as $purchaseOrder) {
            $this->handleXeroTransactionResponse(
                $purchaseOrder,
                XeroTransaction::TYPE_PURCHASE_ORDER,
                $purchaseOrder->PurchaseOrderID,
                $purchaseOrder->PurchaseOrderNumber,
            );
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function updateOrCreateManualJournals(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }

        $xeroManualJournals = [];
        $accountingTransactionsToLock = [];

        $transactions->each(function (AccountingTransaction $transaction) use (&$xeroManualJournals, &$accountingTransactionsToLock) {

            $lineItems = [];

            $passOk = true;

            $transaction->accountingTransactionLines()->each(
                function (AccountingTransactionLine $accountingTransactionLine) use (&$lineItems, &$passOk) {
                    if (! $taxLine = $this->getTaxForAccountingTransactionLine($accountingTransactionLine)) {
                        $passOk = false;

                        return;
                    }
                    $amount = $accountingTransactionLine->quantity * $accountingTransactionLine->amount;
                    $lineItems[] =
                        XeroJournalLine::from([
                            'Description' => $accountingTransactionLine->description.(($accountingTransactionLine->quantity > 1) ?
                                    ' (Qty: '.round($accountingTransactionLine->quantity).')' : ''),
                            'LineAmount' => $accountingTransactionLine->type == AccountingTransactionLine::TYPE_DEBIT ?
                                $amount : -$amount,
                            'AccountCode' => $accountingTransactionLine->nominalCode->code,
                        ] + $taxLine);
                }
            );
            if (! $passOk) {
                return;
            }

            $lineAmountTypes = match ($transaction->is_tax_included) {
                1 => XeroInvoice::LINE_AMOUNT_TYPE_INCLUSIVE,
                0 => XeroInvoice::LINE_AMOUNT_TYPE_EXCLUSIVE,
                default => XeroInvoice::LINE_AMOUNT_TYPE_NO_TAX,
            };

            $narration = $transaction->reference;
            // TODO: Update underlying reference if i want this... because the narration must match the reference exactly to handle
            //  the response
//            if ($transaction->name) {
//                $narration = $transaction->name.' - '.$narration;
//            }

            $xeroManualJournal = XeroManualJournalDto::from([
                'ManualJournalID' => $transaction->accountingIntegration?->xero_uuid,
                'Narration' => $narration,
                'Date' => Helpers::dateUtcToLocal($transaction->transaction_date)->format('Y-m-d'),
                'Status' => XeroManualJournalDto::STATUS_POSTED,
                'LineAmountTypes' => $lineAmountTypes,
                'Url' => $this->getUrl($transaction->id),
                'JournalLines' => $lineItems,
            ]);

            Log::channel('xero')->debug('Xero transaction: '.$transaction->id.' '.$xeroManualJournal->Narration);
            Log::channel('xero')->debug('Xero transaction needs updating');

            /*
             * For new manual journals transactions that will be created in this batch, we need to lock the accounting
             * transaction so that it cannot be retried until it is unlocked through a successful response.  This is
             * to prevent duplicate manual journals being created in Xero.
             */
            //if ((count($xeroManualJournals) < $this->batchSizeManualJournal) && $transaction->accountingIntegration) {
            if ((count($xeroManualJournals) < $this->batchSizeManualJournal)) {
                $accountingTransactionsToLock[] = $transaction->id;
            }

            $xeroManualJournals[] = $xeroManualJournal;
        });

        if (empty($xeroManualJournals)) {
            return;
        }

        Log::channel('xero')->debug(count($xeroManualJournals).' manual journals remaining after removing unnecessary updates');

        $xeroManualJournals = array_slice($xeroManualJournals, 0, $this->batchSizeManualJournal);

        $this->accountingTransactionRepository->setSubmissionStarted($accountingTransactionsToLock);

        $requestManualJournals = XeroManualJournals::from([
            'ManualJournals' => $xeroManualJournals,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseManualJournals = $client->updateOrCreateManualJournals($requestManualJournals->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->take($this->batchSizeManualJournal)->each(function (AccountingTransaction $accountingTransaction) use ($e) {
                $accountingTransaction->saveLastSyncError(Arr::wrap('Connection error: '.$e->getMessage()));
            });

            return;
        }

        /** @var XeroManualJournalDto $journal */
        foreach ($responseManualJournals as $journal) {
            $this->handleXeroTransactionResponse(
                $journal,
                XeroTransaction::TYPE_JOURNAL,
                $journal->ManualJournalID,
                $journal->Narration,
            );
        }
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function updateOrCreateCreditNotes(Collection $transactions)
    {
        if (! $transactions->count()) {
            return;
        }
        $xeroCreditNotes = [];

        $transactions->take(100)->each(
            function (AccountingTransaction $transaction) use (&$xeroCreditNotes) {

                $xeroContactDTO = XeroContactDTO::from([
                    'Name' => $transaction->name,
                ]);

                $lineItems = [];

                $passOk = true;
                $transaction->accountingTransactionLines()->each(
                    function (AccountingTransactionLine $accountingTransactionLine) use (&$lineItems, $transaction, &$passOk) {
                        if (! $taxLine = $this->getTaxForAccountingTransactionLine($accountingTransactionLine)) {
                            $passOk = false;

                            return;
                        }
                        $lineItems[] = XeroLineItems::from([
                            'Description' => $accountingTransactionLine->description,
                            'Quantity' => $accountingTransactionLine->quantity,
                            'UnitAmount' => $accountingTransactionLine->amount +
                            ($transaction->is_tax_included && $accountingTransactionLine->quantity != 0 ?
                                ($accountingTransactionLine->tax_amount / $accountingTransactionLine->quantity) : 0),
                            'AccountCode' => $accountingTransactionLine->nominalCode->code,
                        ] + $taxLine);
                    }
                );
                if (! $passOk) {
                    return;
                }

                $this->handleRounding($lineItems);

                $lineAmountTypes = match ($transaction->is_tax_included) {
                    1 => XeroInvoice::LINE_AMOUNT_TYPE_INCLUSIVE,
                    0 => XeroInvoice::LINE_AMOUNT_TYPE_EXCLUSIVE,
                    default => XeroInvoice::LINE_AMOUNT_TYPE_NO_TAX,
                };

                $xeroCreditNote = XeroCreditNote::from([
                    'Type' => XeroCreditNote::TYPE_ACCRECCREDIT, // we don't have support for supplier credits yet.  Once we do we have to handle this
                    'Contact' => $xeroContactDTO,
                    'Date' => Helpers::dateUtcToLocal($transaction->transaction_date)->format('Y-m-d'),
                    'Status' => XeroInvoice::STATUS_AUTHORISED,
                    'LineAmountTypes' => $lineAmountTypes,
                    'LineItems' => $lineItems,
                    'CurrencyCode' => $transaction->currency_code,
                    'CurrencyRate' => $transaction->currency_rate == 0 ? 0 : (1 / $transaction->currency_rate),
                    'CreditNoteNumber' => $transaction->reference,
                ]);

                if ($parent = $transaction->parent) {
                    $xeroCreditNote->Reference = $parent->reference;
                }

                $xeroCreditNotes[] = $xeroCreditNote;
            }
        );

        if (empty($xeroCreditNotes)) {
            return;
        }

        $requestCreditNotes = XeroCreditNotes::from([
            'CreditNotes' => $xeroCreditNotes,
        ]);

        $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

        try {
            $responseCreditNotes = $client->updateOrCreateCreditNotes($requestCreditNotes->toArray());
        } catch (GuzzleException|ApiException $e) {
            $transactions->take(100)->each(function (AccountingTransaction $accountingTransaction) use ($e) {
                $accountingTransaction->saveLastSyncError(Arr::wrap('Connection error: '.$e->getMessage()));
            });

            return;
        }

        /** @var XeroCreditNote $creditNote */
        foreach ($responseCreditNotes as $creditNote) {
            $this->handleXeroTransactionResponse(
                $creditNote,
                $creditNote->Type,
                $creditNote->CreditNoteID,
                $creditNote->CreditNoteNumber,
            );
        }
    }

    public function allocateXeroTransactions(Collection $transactions): void
    {
        if (! $transactions->count()) {
            return;
        }

        $transactions->take(100)->each(
        /**
         * @throws Throwable
         */ function (AccountingTransaction $transaction) {
                $xeroCreditNote = XeroCreditNoteAllocation::from([
                    'Total' => $transaction->total,
                    'XeroInvoice' => XeroInvoice::from([
                        'InvoiceID' => $transaction->parent?->accountingIntegration?->xero_uuid,
                    ])->only('InvoiceID'),
                ]);

                $client = new Client(app(IntegrationInstanceRepository::class)->getAccountingInstance());

                try {
                    $responseCreditNotes = $client->allocateCreditNotes($xeroCreditNote->toArray(), $transaction->accountingIntegration->xero_uuid);
                } catch (GuzzleException|ApiException $e) {
                    $transaction->saveLastSyncError(Arr::wrap('Connection error: '.$e->getMessage()));
                    return;
                }

                /** @var XeroCreditNote $creditNote */
                foreach ($responseCreditNotes as $creditNote) {
                    DB::transaction(function () use ($creditNote, $transaction) {

                        $has_error = ($creditNote->StatusAttributeString && $creditNote->StatusAttributeString != 'OK') || ! empty((array) $creditNote->ValidationErrors);

                        /** @var XeroTransaction $xeroTransaction */
                        $xeroTransaction = XeroTransaction::query()->firstOrNew([
                            'id' => $transaction->accounting_integration_id ?? null,
                        ]);

                        $json_object = $xeroTransaction->json_object;
                        $json_object['Allocations'] = $creditNote->json_object;

                        if ($has_error) {
                            $transaction->last_sync_error = array_unique(Arr::flatten($creditNote->ValidationErrors));
                            $xeroTransaction->last_error = array_unique(Arr::flatten($creditNote->ValidationErrors));
                            $xeroTransaction->json_object = $json_object;
                            $xeroTransaction->timestamps = false;
                        } else {
                            $transaction->last_sync_error = null;
                            $xeroTransaction->last_error = null;
                            $xeroTransaction->json_object = $json_object;
                        }
                        $xeroTransaction->save();

                        $transaction->allocated_at = Carbon::now();
                        $transaction->timestamps = false;
                        $transaction->save();
                    });
                }
            }
        );
    }

    public function getUrl(int $id): string
    {
        return config('app.env') != 'local' ? config('app.url').'/accounting/transactions?SKUTableAccountingTransactions=f:id:=:'.$id : 'https://www.sku.io';
    }

    private function handleRounding(array &$lineItems): void
    {
        $skuTotal = 0;
        $xeroTotal = 0;

        /** @var XeroLineItems $lineItem */
        foreach ($lineItems as $lineItem) {
            $skuTotal += $lineItem->UnitAmount * $lineItem->Quantity;
            $xeroTotal += round($lineItem->UnitAmount * $lineItem->Quantity, 2);
        }

        if (($rounding_amount = $skuTotal - $xeroTotal) != 0) {
            if (round($rounding_amount, 2) == 0) {
                return;
            }
            $lineItems[] = XeroLineItems::from([
                'Description' => 'Rounding',
                'Quantity' => 1,
                'UnitAmount' => round($rounding_amount, 4),
                'AccountCode' => $this->nominalCodeRepository->getCodeById(Helpers::setting(Setting::KEY_NC_MAPPING_ROUNDING)),
            ]);
        }
    }

    /**
     * @throws Throwable
     */
    private function getTaxForAccountingTransactionLine(AccountingTransactionLine $accountingTransactionLine): ?array
    {
        if (! $accountingTransactionLine->tax_rate_id && is_null($accountingTransactionLine->tax_amount)) {
            return ['TaxType' => 'NONE'];
        } elseif ($accountingTransactionLine->tax_rate_id) {
            $taxRate = $this->xeroTaxRateRepository->getXeroTaxRate($accountingTransactionLine->tax_rate_id);
            if (! $taxRate) {
                $accountingTransactionLine->accountingTransaction->saveLastSyncError(['The tax rate '.$accountingTransactionLine->tax_rate_id.' for the accounting transaction line '.$accountingTransactionLine->id.' is not mapped to a Xero Transaction']);
                return null;
            }
            //            throw_if(
            //                !$taxRate,
            //                new Exception(
            //                    'The tax rate for this accounting transaction is not mapped to a Xero Transaction'
            //                )
            //            );

            return ['TaxType' => $taxRate->TaxType];
        } else {
            return ['TaxAmount' => $accountingTransactionLine->tax_amount];
        }
    }

    /**
     * @throws Exception
     */
    private function getTaxType(AccountingTransactionLine $accountingTransactionLine): string
    {
        $transaction = $accountingTransactionLine->accountingTransaction;

        if ($transaction->tax_rate_id) {
            $taxType = $this->xeroTaxRateRepository->getXeroTaxRate($transaction->tax_rate_id)->TaxType;
        } elseif ($accountingTransactionLine->tax_rate_id) {
            if (! $xeroTaxRate = $this->xeroTaxRateRepository->getXeroTaxRate($accountingTransactionLine->tax_rate_id)) {
                throw new Exception(
                    'The tax rate for this accounting transaction is not mapped to a Xero Transaction'
                );
            }
            $taxType = $xeroTaxRate->TaxType;
        } else {
            $taxType = 'NONE';
        }

        return $taxType;
    }

    private function sanitizeValidationErrors(XeroCreditNote|XeroInvoice $response, ?array $request = []): ?array
    {
        /*
         * Check if xero validation errors are entirely due to a locked transaction, if not, return back validation
         * errors
         */
        if ($this->errorsAreDueToLockedTransaction($response->ValidationErrors)) {
            // If so, check return the array of payload needing to be updated, or if no update needed, return null
            return $this->updatableTransactionPayload($response, $request) ?? null;
        } else {
            return $response->ValidationErrors;
        }
    }

    private function errorsAreDueToLockedTransaction(array|Optional $ValidationErrors): bool
    {
        return collect($ValidationErrors)->pluck('Message')->every(fn ($error) => in_array($error, XeroLockedTransactionErrorEnum::values()));
    }

    private function updatableTransactionPayload(XeroCreditNote|XeroInvoice $response, ?array $request = []): ?array
    {
        // For now, we only look for differences in line items.
        if (empty($request) || ! isset($request['LineItems']) || count($request['LineItems']) != count($response->LineItems)) {
            $entity = str(str($response::class)->explode('\\')->last())->snake()->replace('_', ' ')->title();
            $details = 'Entity: '.$entity.' ID: '.$response->getId();
            throw new InvalidArgumentException('Request and response line items mismatch: '.$details);
        }

        $differences = [];

        foreach ($request['LineItems'] as $key => $requestItem) {
            $requestItem = $requestItem instanceof Data ? $requestItem->toArray() : $requestItem;
            $responseItem = $response->LineItems[$key];
            array_walk($responseItem, function ($item, $key) use (&$responseItem, &$requestItem) {
                if (is_numeric($item) && ! is_int($item)) {
                    $responseItem[$key] = number_format($item, 4);
                    $requestItem[$key] = number_format($item, 4);
                }
            });

            $diff = Helpers::arrayDiffAssoc2($requestItem, $responseItem);

            if (! empty($diff)) {
                foreach ($diff as $field => $value) {
                    $localValue = (@$requestItem[$field] ?: '');
                    $serverValue = (@$response->LineItems[$key][$field] ?: '');
                    $differences[] = "Field $field has Local Value: '$localValue', Server Value: '$serverValue'";
                }
            }
        }

        if (empty($differences)) {
            return null;
        }

        return ['Xero Transaction Locked, but needs update: '.rtrim(implode(", \n", $differences))];
    }

    public function transactionNeedsUpdate(array $requestLineItems, ?array $xeroLineItems, bool $hasQty = true): bool
    {
        if (! $xeroLineItems) {
            return true;
        }

        $requestLineItems = array_map(function ($lineItem) use ($hasQty) {
            $data = [
                'AccountCode' => $lineItem->AccountCode ?? null,
                'Description' => $lineItem->Description,
            ];
            if ($hasQty) {
                $data['Quantity'] = $lineItem->Quantity;
            }
            if (isset($lineItem->UnitAmount)) {
                $data['UnitAmount'] = round($lineItem->UnitAmount, 2);
            } elseif (isset($lineItem->LineAmount)) {
                $data['LineAmount'] = round($lineItem->LineAmount, 2);
            }

            return $data;
        }, $requestLineItems);

        $xeroLineItems = array_map(function ($lineItem) use ($hasQty) {
            $data = [
                'AccountCode' => $lineItem['AccountCode'] ?? null,
                'Description' => $lineItem['Description'],
            ];
            if ($hasQty) {
                $data['Quantity'] = $lineItem['Quantity'];
            }
            if (isset($lineItem['UnitAmount'])) {
                $data['UnitAmount'] = round($lineItem['UnitAmount'], 2);
            } elseif (isset($lineItem['LineAmount'])) {
                $data['LineAmount'] = round($lineItem['LineAmount'], 2);
            }

            return $data;
        }, $xeroLineItems);

        if ($requestLineItems != $xeroLineItems) {
            Log::channel('xero')->debug('Request line items do not match xero line items', [
                'requestLineItems' => $requestLineItems,
                'xeroLineItems' => $xeroLineItems,
                'diff' => arrayRecursiveDiff($requestLineItems, $xeroLineItems),
            ]);

            return true;
        }

        return false;
    }

    public function getLastModifiedJournalDate(): ?Carbon
    {
        $lastModified = XeroManualJournal::query()
            ->orderBy('UpdatedDateUTC', 'desc')
            ->first()
            ->UpdatedDateUTC;

        return $lastModified ? Carbon::parse($lastModified)->addSecond() : null;
    }

    public function saveJournal(XeroManualJournalDto $xeroManualJournalDto): void
    {
        if (! XeroManualJournal::query()->where('ManualJournalID', $xeroManualJournalDto->ManualJournalID)->exists()) {
            $xeroManualJournal = new XeroManualJournal();
            $xeroManualJournal->ManualJournalID = $xeroManualJournalDto->ManualJournalID;
            $xeroManualJournal->DateUTC = Helpers::unixTimestampToCarbon($xeroManualJournalDto->Date);
            $xeroManualJournal->UpdatedDateUTC = Helpers::unixTimestampToCarbon($xeroManualJournalDto->json_object['UpdatedDateUTC']);
            $xeroManualJournal->json_object = $xeroManualJournalDto->json_object;
            $xeroManualJournal->save();
        }
    }

    public function saveBulkJournal(XeroManualJournalCollection $data): void
    {
        $data->chunk(25000)->each(function ($chunk) {
            $chunk = $chunk->map(function (XeroManualJournalDto $xeroManualJournalDto) {
                $xeroManualJournalDto->DateUTC = Helpers::unixTimestampToCarbon($xeroManualJournalDto->Date);
                $xeroManualJournalDto->UpdatedDateUTC = Helpers::unixTimestampToCarbon($xeroManualJournalDto->json_object['UpdatedDateUTC']);

                return $xeroManualJournalDto;
            });

            XeroManualJournal::dataFeedBulkImport([
                'data_to_import' => [
                    'type' => 'json',
                    'data' => json_encode($chunk),
                ],
                'insert' => true,
                'update' => true,
                'mappings' => [
                    [
                        'expected_column_name' => 'ManualJournalID',
                        'data_column_name' => 'ManualJournalID',
                        'expected_column_type' => 'string',
                    ],
                    [
                        'expected_column_name' => 'DateUTC',
                        'data_column_name' => 'DateUTC',
                        'expected_column_type' => 'string',
                    ],
                    [
                        'expected_column_name' => 'UpdatedDateUTC',
                        'data_column_name' => 'UpdatedDateUTC',
                        'expected_column_type' => 'string',
                    ],
                    [
                        'expected_column_name' => 'json_object',
                        'data_column_name' => 'json_object',
                        'expected_column_type' => 'string',
                    ],
                ],
                'unique_by_columns' => [
                    'ManualJournalID',
                ],
                'default_columns' => [
                    [
                        'expected_column_name' => 'created_at',
                        'default_value' => now()->toDateTimeString(),
                        'expected_column_type' => 'datetime',
                    ],
                    [
                        'expected_column_name' => 'updated_at',
                        'default_value' => now()->toDateTimeString(),
                        'expected_column_type' => 'datetime',
                    ],
                ],
            ]);
        });
    }

    public function existingXeroTransactionHasError(?XeroTransaction $xeroTransaction): bool
    {
        if (! $xeroTransaction) {
            return false;
        }

        if (! isset($xeroTransaction->json_object['StatusAttributeString'])) {
            return false;
        }

        return $xeroTransaction->json_object['StatusAttributeString'] === 'ERROR';
    }
}
