<?php

namespace Modules\Xero\Managers;

use App\Exceptions\ApiException;
use App\Models\AccountingTransaction;
use App\Models\IntegrationInstance;
use App\Models\Supplier;
use App\Repositories\IntegrationInstanceRepository;
use App\Repositories\NominalCodeRepository;
use Carbon\Carbon;
use Exception;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Arr;
use Modules\Xero\DTO\XeroAccount as XeroAccountDto;
use Modules\Xero\DTO\XeroAddress;
use Modules\Xero\DTO\XeroContact as XeroContactDTO;
use Modules\Xero\DTO\XeroContacts;
use Modules\Xero\DTO\XeroManualJournalDto;
use Modules\Xero\DTO\XeroPhone;
use Modules\Xero\DTO\XeroTaxRate;
use Modules\Xero\Entities\XeroAccount;
use Modules\Xero\Entities\XeroIntegrationInstance;
use Modules\Xero\Entities\XeroManualJournal;
use Modules\Xero\Exceptions\XeroDailyApiLimitReachedException;
use Modules\Xero\Jobs\XeroDeletePaymentsJob;
use Modules\Xero\Jobs\XeroUpdateOrCreatePaymentsJob;
use Modules\Xero\Repositories\XeroAccountRepository;
use Modules\Xero\Repositories\XeroContactRepository;
use Modules\Xero\Repositories\XeroPaymentRepository;
use Modules\Xero\Repositories\XeroTaxRateRepository;
use Modules\Xero\Repositories\XeroTransactionRepository;
use Modules\Xero\Services\Client;
use Spatie\DataTransferObject\Exceptions\UnknownProperties;

/*
 * TODO: Need to handle Goods Invoiced Not Received and Goods Received Not Invoiced for partial receipts & partial invoices
 * TODO: Handle deletions/voiding
 */

class XeroManager
{
    /** @var Client The Xero Client */
    private Client $client;

    private XeroIntegrationInstance|IntegrationInstance $integrationInstance;

    /**
     * @throws Exception
     */
    public function __construct(
        private readonly XeroTransactionRepository $transactions,
        private readonly XeroPaymentRepository $xeroPaymentRepository,
        private readonly XeroTaxRateRepository $taxRates,
        private readonly XeroContactRepository $contacts,
        private readonly XeroAccountRepository $accounts,
        private readonly NominalCodeRepository $nominalCodes,
    ) {

        $integrationInstance = app(IntegrationInstanceRepository::class)->getAccountingInstance();
        if ($integrationInstance instanceof IntegrationInstance) {
            $this->integrationInstance = $integrationInstance;
        } else {
            return;
        }
        $this->client = new Client($this->integrationInstance);
    }

    /**
     * @throws UnknownProperties|Exception
     */
    public function syncPayments(array $payment_ids = []): void
    {
        $this->xeroPaymentRepository->getPaymentsNeedingDelete($payment_ids)
            ->chunk(100)->each(function (Collection $paymentsToDelete) {
                dispatch(new XeroDeletePaymentsJob($paymentsToDelete))->onQueue('xero');
            });

        $this->xeroPaymentRepository->getPaymentsNeedingUpdate($payment_ids)
            ->chunk(100)->each(function (Collection $paymentsToUpdateOrCreate) {
                dispatch(new XeroUpdateOrCreatePaymentsJob($paymentsToUpdateOrCreate))->onQueue('xero');
            });
    }

    /**
     * @throws GuzzleException
     * @throws ApiException
     */
    public function getTransactions(array $accounting_transaction_ids = []): void
    {
        //$xero_uuids = $this->xeroTransactionRepository->getXeroUuidsFromAccountingTransactionIds($accounting_transaction_ids);
        $xero_uuids = $this->transactions->getXeroUuidsForErroredTransactions();
        $this->transactions->getTransactions($xero_uuids);
    }

    /**
     * @throws XeroDailyApiLimitReachedException
     * @throws Exception
     */
    public function syncTransactions(array $accounting_transaction_ids = []): void
    {
        if (! $this->integrationInstance->hasRemainingDailyApiCalls()) {
            throw new XeroDailyApiLimitReachedException('Xero Daily API calls limit reached');
        }
        $this->transactions->purgeDeletedTransactions();

        $transactions = $this->transactions->getTransactionsNeedingVoid($accounting_transaction_ids);
        if ($transactions->count()) {
            $this->transactions->voidXeroTransactions($transactions);
        }

        if ($accounting_transaction_ids) {
            $transactions = $this->transactions->getForValues($accounting_transaction_ids, 'id', AccountingTransaction::class);
        } else {
            $transactions = $this->transactions->getTransactionsNeedingUpdate($accounting_transaction_ids);
        }
        // Don't sync any transactions with an error, they must be manually cleared first
        // Also don't sync transactions that have been locked due to submission already started
        $transactions = $transactions->reject(function (AccountingTransaction $transaction) {
            return $transaction->last_sync_error || $transaction->submission_started_at;
        });

        if ($transactions->count()) {
            $this->transactions->updateXeroTransactions($transactions);
        }

        /*
         * TODO: Commenting this out for now until refactor of Sales Credits / Payments.  When this is done, has to be
         *  carefully considered for the proper use case (i.e. don't allocate for refunds)
         */
        /*$transactions = $this->xeroTransactionRepository->getTransactionsNeedingAllocation($accounting_transaction_ids);
        if ($transactions->count())
        {
            $this->xeroTransactionRepository->allocateXeroTransactions($transactions);
        }*/
    }

    /**
     * @throws Exception
     */
    public function syncSuppliers(array $supplier_ids = []): void
    {
        if (! $suppliersNeedingUpdate = $this->contacts->getSuppliersNeedingUpdate($supplier_ids)) {
            return;
        }

        [$contacts, $suppliersMap] = $this->makeContacts($suppliersNeedingUpdate);

        if (! ($contacts)) {
            return;
        }

        // Sync the contacts with Xero
        try {
            $contacts = $this->client->updateOrCreateContacts($contacts->toArray());
        } catch (GuzzleException|ApiException) {
            // Indicate that there were errors with the sync on the Xero contacts.
            $suppliersNeedingUpdate->each(function (Supplier $supplier) {
                $this->contacts->syncContact([
                    'link_id' => $supplier->id,
                    'link_type' => Supplier::class,
                    'last_error' => Arr::wrap('Connection error'),
                ]);
            });

            return;
        }

        $this->contacts->updateXeroSuppliers($contacts, $suppliersMap);

        // Indicate the update on the integration instance.
        $settings = $this->integrationInstance->integration_settings;
        $settings['lastSyncContacts'] = Carbon::now();
        $this->integrationInstance->integration_settings = $settings;
        $this->integrationInstance->update();
    }

    /**
     * @return array<XeroContacts, array>|null
     */
    private function makeContacts(Collection $suppliers): ?array
    {
        if (! $suppliers->count()) {
            return null;
        }
        $xeroContacts = [];
        $suppliersMap = [];

        $suppliers->each(function (Supplier $supplier) use (&$xeroContacts, &$suppliersMap) {

            $xeroAddress = XeroAddress::from([
                XeroAddress::ADDRESS_TYPE_BILLING,
                $supplier->address?->address1,
                $supplier->address?->address2,
                $supplier->address?->address3,
                $supplier->address?->city,
                $supplier->address?->zip,
                $supplier->address?->country,
                $supplier->address?->name,
            ]);

            $xeroPhone = XeroPhone::from([
                'PhoneType' => XeroPhone::PHONE_TYPE_DEFAULT,
                'PhoneNumber' => $supplier->phone,
            ]);

            $xeroContacts[] = XeroContactDTO::from([
                'Name' => $supplier->name,
                'EmailAddress' => $supplier->email,
                'Addresses' => Arr::wrap($xeroAddress),
                'Phones' => Arr::wrap($xeroPhone),
            ]);
            $suppliersMap[] = $supplier->id;
        });

        return [
            XeroContacts::from([
                'Contacts' => $xeroContacts,
            ]),
            $suppliersMap,
        ];
    }

    /*
     * TODO: move more logic here and translate between our data layer and 3rd party data layer (routing/mapping)
     */

    public function createNominalCodesFromAccounts(array $ids): void
    {
        $this->nominalCodes->createMany(
            $this->accounts->getAccountsWithoutNominalCodesIn($ids)
                ->map(fn (XeroAccount $account) => [
                    'code' => $account->Code,
                    'name' => $account->Name,
                    'type' => XeroAccount::TYPE_NOMINAL_CODE_MAPPINGS[$account->Type],
                ])
                ->toArray()
        );
    }

    /**
     * @throws ApiException
     */
    public function downloadAccounts(): void
    {
        $this->client->getAccounts()->each(function (XeroAccountDto $account) {
            $this->accounts->syncAccount($account);
        });
    }

    /**
     * @throws ApiException
     */
    public function downloadManualJournals(): void
    {
        $modifiedAfter = $this->transactions->getLastModifiedJournalDate();

        $manualJournals = $this->client->getManualJournals($modifiedAfter);

        if (count($manualJournals) > XeroManualJournal::BULK_THRESHOLD) {
            $this->transactions->saveBulkJournal($manualJournals);
        } else {
            $manualJournals->each(function (XeroManualJournalDto $manualJournal) {
                $this->transactions->saveJournal($manualJournal);
            });
        }
    }

    /**
     * @throws ApiException
     * @throws GuzzleException
     */
    public function downloadTaxRates(): void
    {
        $this->client->getTaxRates()->each(function (XeroTaxRate $taxRate) {
            $this->taxRates->syncTaxRate($taxRate);
        });
    }

    /**
     * @throws Exception
     */
    public function deletePayments(array $payment_ids): void
    {
        $payments = $this->xeroPaymentRepository->getFromPaymentIds($payment_ids);
        $this->xeroPaymentRepository->deleteXeroPayments($payments);
    }
}
