<?php

namespace Modules\Xero\Http\Controllers;

use App\DataTable\DataTable;
use App\Exceptions\ApiException;
use App\Http\Controllers\Controller;
use App\Repositories\IntegrationInstanceRepository;
use App\Response;
use Illuminate\Http\Request;
use Modules\Xero\Entities\XeroAccount;
use Modules\Xero\Exceptions\XeroReauthorizeException;
use Modules\Xero\Exceptions\XeroTimeoutException;
use Modules\Xero\Managers\XeroManager;
use Modules\Xero\Transformers\XeroAccountResource;

class XeroAccountController extends Controller
{
    use DataTable;

    public function __construct(
        private readonly XeroManager $manager,
        private readonly IntegrationInstanceRepository $integrationInstances,
    ) {
        parent::__construct();
    }

    public function download(): ?Response
    {
        try {
            $this->manager->downloadAccounts();

            return $this->response->success(message: 'Xero Accounts downloaded successfully.');
        } catch (ApiException $e) {
            return $this->response->addError(
                'There was a problem downloading Xero Accounts: '.$e->getMessage(),
                500,
                'DownloadXeroAccounts'
            );
        } catch (XeroReauthorizeException $e) {
            return $this->response->addError(
                'Xero authorization has expired and must be reauthorized.',
                'XeroReauthorize',
                'XeroReauthorize'
            );
        } catch (XeroTimeoutException $e) {
            return $this->response->addError(
                $e->getMessage(),
                'XeroTimeout',
                'XeroTimeout'
            );
        }
    }

    public function createNominalCodes(Request $request): Response
    {
        $validated = $request->validate([
            'ids' => 'required|array|min:1',
        ]);

        // We ensure that there is an accounting integration.
        if (! ($instance = $this->integrationInstances->getAccountingInstance()) || ! $instance->isXero()) {
            return $this->response->addError('No Xero accounting integration found', 404, 'message')->error(404);
        }

        $this->manager->createNominalCodesFromAccounts($validated['ids']);

        return $this->response->success(200);
    }

    public function getModel(): string
    {
        return XeroAccount::class;
    }

    public function getResource(): string
    {
        return XeroAccountResource::class;
    }
}
