<?php

namespace Modules\Xero\Entities;

use App\Models\AccountingTransaction;
use App\Models\Concerns\HasFilters;
use App\Models\Contracts\Filterable;
use App\Services\Accounting\AccountingIntegration;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\MorphOne;

/**
 * @property int $id
 * @property string $xero_uuid
 * @property string $type,
 * @property ?array $last_error,
 * @property array $json_object
 * @property array $request_body
 * @property string $link
 * @property AccountingTransaction $accountingTransaction
 * @property Carbon $last_synced_at
 * @property Carbon $updated_at
 * @property Carbon $created_at
 */
class XeroTransaction extends Model implements AccountingIntegration, Filterable
{
    use HasFilters;

    const TYPE_ACCREC = 'ACCREC';

    const TYPE_ACCPAY = 'ACCPAY';

    const TYPE_ACCRECCREDIT = 'ACCRECCREDIT';

    const TYPE_JOURNAL = 'JOURNAL';

    const TYPE_PURCHASE_ORDER = 'PURCHASE_ORDER';

    const TYPES_INVOICE = [
        AccountingTransaction::TYPE_SALES_ORDER_INVOICE => self::TYPE_ACCREC,
        AccountingTransaction::TYPE_BATCH_SALES_ORDER_INVOICE => self::TYPE_ACCREC,
        AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE => self::TYPE_ACCPAY,
    ];

    protected $fillable = [
        'xero_uuid',
        'type',
        'last_error',
        'json_object',
        'request_body',
        'updated_at',
    ];

    protected $casts = [
        'last_error' => 'array',
        'json_object' => 'array',
        'request_body' => 'array',
    ];

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function accountingTransaction(): MorphOne
    {
        return $this->morphOne(AccountingTransaction::class, 'accounting_integration');
    }

    public function getLinkAttribute()
    {
        switch ($this->type) {
            case 'ACCREC':
                return 'https://go.xero.com/AccountsReceivable/View.aspx?InvoiceID='.$this->json_object['InvoiceID'];
            case 'ACCPAY':
                return 'https://go.xero.com/AccountsPayable/View.aspx?InvoiceID='.$this->json_object['InvoiceID'];
            case 'ACCRECCREDIT':
                return 'https://go.xero.com/AccountsReceivable/ViewCreditNote.aspx?creditNoteID='.$this->json_object['CreditNoteID'];
            case 'JOURNAL':
                return 'https://go.xero.com/Journal/View.aspx?invoiceID='.$this->json_object['ManualJournalID'];
            case 'PURCHASE_ORDER':
                return 'https://go.xero.com/Accounts/Payable/PurchaseOrders/View/'.$this->json_object['PurchaseOrderID'];
        }
    }

    public function lastSyncedAt(): Attribute
    {
        return Attribute::get(fn () => $this->updated_at);
    }

    public function getLastSyncedAt(): ?Carbon
    {
        return $this->last_synced_at;
    }

    public function getLastError(): ?array
    {
        return $this->last_error;
    }

    public function getLink(): ?string
    {
        return $this->link;
    }

    public function delete(): ?bool
    {
        if ($this->accountingTransaction) {
            $this->accountingTransaction->update([
                'accounting_integration_type' => null,
                'accounting_integration_id' => null,
            ]);
        }

        return parent::delete();
    }

    public function filterableColumns(): array
    {

        return collect($this->availableColumns())->where('filterable', 1)->pluck('data_name')->all();

    }

    public function availableColumns()
    {
        return [];
    }

    public function generalFilterableColumns(): array
    {
        return $this->filterableColumns();

    }
}
