<?php

namespace Modules\WooCommerce\Tests\Feature\Controllers;

use App\Data\CreateSkuProductsFromSalesChannelData;
use App\Data\SalesChannelProductImportMappingData;
use App\Data\SalesChannelProductToSkuProductMappingCollectionData;
use App\Data\SalesChannelProductToSkuProductMappingData;
use App\Jobs\ImportSalesChannelProductMappingsJob;
use App\Jobs\MapSalesOrderLinesToSalesChannelProductsJob;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\User;
use Exception;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetProductsAdt;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Entities\WooCommerceOrder;
use Modules\WooCommerce\Entities\WooCommerceOrderItem;
use Modules\WooCommerce\Entities\WooCommerceProduct;
use Modules\WooCommerce\Jobs\RefreshWooCommerceProductsJob;
use Modules\WooCommerce\Managers\WooCommerceProductManager;
use Modules\WooCommerce\Tests\WooCommerceMockRequests;
use Modules\WooCommerce\Tests\WooCommerceTestingData;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class WooCommerceProductControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WooCommerceMockRequests;

    private WooCommerceIntegrationInstance $wooCommerceIntegrationInstance;

    public function setUp(): void
    {
        parent::setUp();
        $this->wooCommerceIntegrationInstance = WooCommerceIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        Queue::fake();
        Sanctum::actingAs(User::first());
    }

    /**
     * @throws Exception
     * @throws Throwable
     */
    public function test_woo_commerce_products_controller(): void
    {
        /*
        |--------------------------------------------------------------------------
        | Refresh products
        |--------------------------------------------------------------------------
        */

        $this->mockRefreshProducts();

        $this->postJson(route('woo-commerce.products.refresh', $this->wooCommerceIntegrationInstance->id))->assertOk();

        Queue::assertPushed(RefreshWooCommerceProductsJob::class);

        (new WooCommerceProductManager($this->wooCommerceIntegrationInstance))->refreshProducts(new WooCommerceGetProductsAdt());

        $this->assertDatabaseHas((new WooCommerceProduct())->getTable(), [
            'integration_instance_id' => $this->wooCommerceIntegrationInstance->id,
            'json_object' => json_encode(WooCommerceTestingData::getProduct()),
        ]);

        /*
        |--------------------------------------------------------------------------
        | Get products
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('woo-commerce.products.index', $this->wooCommerceIntegrationInstance->id))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                '*' => [
                    'id',
                    'woo_commerce_id',
                    'sku',
                    'name',
                    'slug',
                    'type',
                    'status',
                    'price',
                    'weight',
                    'dimensions_length',
                    'dimensions_width',
                    'dimensions_height',
                    'featured',
                    'catalog_visibility',
                    'date_created_gmt',
                    'date_modified_gmt',
                    'product',

                ],
            ],
        ]);

        $firstProductId = $response->json('data')[0]['id'];
        $firstProductUniqueId = $response->json('data')[0]['woo_commerce_id'];
        $firstProductName = $response->json('data')[0]['name'];
        $firstProductSku = $response->json('data')[0]['sku'];

        /*
        |--------------------------------------------------------------------------
        | Show product
        |--------------------------------------------------------------------------
        */

        $response = $this->getJson(route('woo-commerce.products.show', [$this->wooCommerceIntegrationInstance->id, $firstProductId]))->assertOk();

        $response->assertJsonStructure([
            'data' => [
                'id',
                'integration_instance_id',
                'woo_commerce_id',
                'sku',
                'name',
                'slug',
                'type',
                'status',
                'price',
                'weight',
                'dimensions_length',
                'dimensions_width',
                'dimensions_height',
                'featured',
                'catalog_visibility',
                'date_created_gmt',
                'date_modified_gmt',
                'json_object',
                'created_at',
                'updated_at',
            ],
        ]);

        /*
        |--------------------------------------------------------------------------
        | Create sku products
        |--------------------------------------------------------------------------
        */

        $this->postJson(route('woo-commerce.products.create-sku-products', $this->wooCommerceIntegrationInstance->id), CreateSkuProductsFromSalesChannelData::from([
            'create_all_products' => true,
        ])->toArray())->assertOk();

        $product = Product::first();

        $this->assertDatabaseHas((new Product())->getTable(), [
            'id' => $product->id,
            'sku' => $firstProductSku,
        ]);

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->wooCommerceIntegrationInstance->salesChannel->id,
            'document_type' => WooCommerceProduct::class,
            'document_id' => $firstProductId,
            'listing_sku' => $firstProductSku,
            'product_id' => $product->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Unmap product
        |--------------------------------------------------------------------------
        */

        $this->putJson(route('woo-commerce.products.map', $this->wooCommerceIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $firstProductUniqueId,
                    'mapped_sku' => null,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseEmpty((new ProductListing())->getTable());

        /*
        |--------------------------------------------------------------------------
        | Map products
        |--------------------------------------------------------------------------
        */

        //Data to test mapping of sales order lines.
        $salesChannelOrderLine = WooCommerceOrderItem::factory()->hasOrder()->create()->refresh();

        $salesChannelOrderLine->json_object = array_merge($salesChannelOrderLine->json_object, [
            $salesChannelOrderLine->getSalesChannelProductUniqueId() => $firstProductUniqueId,
        ]);
        $salesChannelOrderLine->save();

        $salesOrder = SalesOrder::factory()->create([
            'sales_channel_order_type' => WooCommerceOrder::class,
            'sales_channel_order_id' => 1,
        ]);

        $salesOrderLine = SalesOrderLine::factory()
            ->for($salesOrder)
            ->create([
                'product_id' => null,
                'product_listing_id' => null,
                'sales_channel_line_id' => $salesChannelOrderLine->{$salesChannelOrderLine::getTableUniqueId()},
            ]);

        $this->putJson(route('woo-commerce.products.map', $this->wooCommerceIntegrationInstance->id), SalesChannelProductToSkuProductMappingCollectionData::from([
            'mapping' => SalesChannelProductToSkuProductMappingData::collection([
                SalesChannelProductToSkuProductMappingData::from([
                    'sales_channel_listing_id' => $firstProductUniqueId,
                    'mapped_sku' => $firstProductSku,
                ]),
            ]),
        ])->toArray())->assertOk();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->wooCommerceIntegrationInstance->salesChannel->id,
            'document_type' => WooCommerceProduct::class,
            'document_id' => $firstProductId,
            'listing_sku' => $firstProductSku,
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);
        (end(Queue::pushedJobs()[MapSalesOrderLinesToSalesChannelProductsJob::class])['job'])->handle();

        $this->assertDatabaseHas((new SalesOrderLine())->getTable(), [
            'id' => $salesOrderLine->id,
            'product_id' => $product->id,
            'product_listing_id' => ProductListing::whereListingSku($firstProductSku)->first()->id,
        ]);

        /*
        |--------------------------------------------------------------------------
        | Export products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('woo-commerce.products.export', $this->wooCommerceIntegrationInstance->id))->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Download CSV products
        |--------------------------------------------------------------------------
        */

        $this->getJson(route('woo-commerce.products.export-download', $this->wooCommerceIntegrationInstance->id), [
            'included' => [
                'woo_commerce_id',
                'name',
                'sku',
                'mapped_sku',
            ],
        ])->assertOk()->assertDownload();

        /*
        |--------------------------------------------------------------------------
        | Upload CSV products
        |--------------------------------------------------------------------------
        */

        $product = Product::factory()->create([
            'sku' => 'NewProduct',
        ]);

        $csvData = [
            [WooCommerceProduct::getUniqueField(), 'Title', 'SKU', 'mapped_sku'],
            [$firstProductUniqueId, $firstProductName, $firstProductSku, 'NewProduct'],
        ];

        $path = Storage::disk('sales-channel-product-mappings')->path('test listings.csv');
        $fh = fopen($path, 'w');

        foreach ($csvData as $row) {
            fputcsv($fh, $row);
        }

        fclose($fh);

        $mappingData = SalesChannelProductImportMappingData::from([
            'original_name' => 'test.csv',
            'stored_name' => 'test listings.csv',
        ]);

        $this->postJson(route('woo-commerce.products.import-mappings', $this->wooCommerceIntegrationInstance->id), $mappingData->toArray())->assertOk();

        Queue::assertPushed(ImportSalesChannelProductMappingsJob::class);

        $manager = new WooCommerceProductManager($this->wooCommerceIntegrationInstance);

        (new ImportSalesChannelProductMappingsJob($mappingData, $manager))->handle();

        $this->assertDatabaseHas((new ProductListing())->getTable(), [
            'sales_channel_id' => $this->wooCommerceIntegrationInstance->salesChannel->id,
            'document_type' => WooCommerceProduct::class,
            'document_id' => $firstProductId,
            'sales_channel_listing_id' => $firstProductUniqueId,
            'listing_sku' => $firstProductSku,
            'product_id' => $product->id,
        ]);

        Queue::assertPushed(MapSalesOrderLinesToSalesChannelProductsJob::class);
    }
}
