<?php

namespace Modules\WooCommerce\Services;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelClientInterface;
use App\Enums\HttpMethodEnum;
use App\Helpers;
use Exception;
use Illuminate\Support\Collection;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceFulfillOrderAdt;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetOrdersAdt;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetProductsAdt;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceUpdateOrderAdt;
use Modules\WooCommerce\Data\WooCommerceOrderData;
use Modules\WooCommerce\Data\WooCommerceProductData;
use Modules\WooCommerce\Data\WooCommerceResponseData;
use Modules\WooCommerce\Repositories\WooCommerceOrderRepository;
use Modules\WooCommerce\Repositories\WooCommerceProductRepository;
use Throwable;

/**
 * NOTE: for any WooCommerce installations its possible to override the default number of items returned per page.
 *       For this reason we cant really rely on this default and instead `per_page` is being included in each request URL
 */
class WooCommerceClient extends WooCommerceAuthenticationClient implements SalesChannelClientInterface
{
    public function __construct(IntegrationInstanceInterface $integrationInstance)
    {
        parent::__construct($integrationInstance);
    }

    public function getOrder(string $orderId): ClientResponseDataInterface
    {
        $response = $this->request(
            HttpMethodEnum::GET,
            '/orders/'.$orderId
        );

        $collection = new Collection();
        $collection->push(WooCommerceOrderData::from($response->json()));

        return WooCommerceResponseData::from([
            'collection' => $collection,
            'nextToken' => null,
        ]);
    }

    /**
     * NOTE: WooCommerce will set both "date_created" and "date_modified" to the current time when an order is created
     *
     * @throws Exception
     */
    public function getOrders(WooCommerceGetOrdersAdt|ApiDataTransformerInterface $parameters): WooCommerceResponseData
    {
        if (is_null($parameters->modified_after)) {
            $parameters->modified_after = app(WooCommerceOrderRepository::class)
                ->getStartDateForNew($this->integrationInstance);
        }

        $response = $this->request(
            HttpMethodEnum::GET,
            '/orders',
            $parameters->transform()
        );

        $collection = WooCommerceOrderData::collection($response->json())->toCollection();

        return WooCommerceResponseData::from([
            'collection' => $collection,
            'nextPage' => Helpers::getPageNumberFromLinkHeader($response->header('Link')),
            'modifiedAfterDateTime' => $parameters->modified_after,
        ]);
    }

    public function getProduct(string $productId): array
    {
        // TODO: Implement getProduct() method.
        return [];
    }

    /**
     * @throws Exception
     */
    public function getProducts(WooCommerceGetProductsAdt|ApiDataTransformerInterface $parameters): WooCommerceResponseData
    {
        if (is_null($parameters->modified_after)) {
            $parameters->modified_after = app(WooCommerceProductRepository::class)
                ->getStartDateForNew($this->integrationInstance);

            if (!is_null($parameters->modified_after)) {
                // TODO: This temporarily disabled retrieving only new products until we figure out how to properly
                //  combine pagination and filtering for woocommerce
                $parameters->modified_after = null;
                //$parameters->modified_after = Carbon::parse($parameters->modified_after)->subSecond();
            }
        }

        $response = $this->request(
            HttpMethodEnum::GET,
            '/products',
            $parameters->transform()
        );

        $collection = WooCommerceProductData::collection($response->json());

        return WooCommerceResponseData::from([
            'collection' => $collection,
            'nextPage' => Helpers::getPageNumberFromLinkHeader($response->header('Link')),
            'modifiedAfterDateTime' => $parameters->modified_after,
        ]);
    }

    /**
     * Unless https://woo.com/document/shipment-tracking/ extension is installed, we can only fulfill a WooCommerce
     * order by marking the order status as completed.
     *
     * TODO: WooCommerce still needs to trigger fulfillOrderForSalesOrder... for WooCommerce this should only be called
     *  once an order is completely fulfilled. @Jatin let's discuss the best place to call this.
     *
     *
     * @throws Throwable
     */
    public function fulfillOrder(WooCommerceFulfillOrderAdt|ApiDataTransformerInterface $parameters): bool
    {
        $requestParams = $parameters->transform();

        $response = $this->request(
            HttpMethodEnum::PUT,
            '/orders/'.$parameters->wooCommerceOrderId,
            $requestParams,
        );

        throw_if(! $response->ok(), 'WooCommerce order update failed for ID '.$parameters->wooCommerceOrderId);

        if ($parameters->trackingNumber) {
            $response = $this->request(
                HttpMethodEnum::POST,
                '/orders/'.$parameters->wooCommerceOrderId . '/notes',
                [
                    'note' => $parameters->trackingNumber,
                ],
            );
            throw_if(! $response->successful(), 'WooCommerce order note update failed for ID '.$parameters->wooCommerceOrderId);
        }

        return true;
    }

    public function fulfillOrders(ApiDataTransformerInterface $parameters): void
    {
        // Not implemented for WooCommerce
    }

    public function syncInventory(ApiDataTransformerInterface $parameters)
    {
        // TODO: Implement syncInventory() method.
    }
}
