<?php

namespace Modules\WooCommerce\Services;

use Carbon\CarbonImmutable;
use Modules\WooCommerce\Services\Exceptions\DateTimeHelperException;

/**
 * NOTE ABOUT WOOCOMMERCE DATE TIME:
 *
 * The date time returned from WooCommerce is in a format like 2022-11-21T14:32:25
 * The WooCommerce API docs state it us using ISO8601, but it seems the API
 * is not accurately using that format. Notice that the timezone is NOT included
 *
 * The WooCommerce API often returns fields such as
 *   date_created
 *   date_modified
 *   date_created_gmt
 *   date_modified_gmt
 *
 * The date_created and date_modified will return a different time which will change if the
 * "Timezone" setting in WordPress is changed. This is why it is better to use date_created_gmt and date_modified_gmt
 */
class DateTimeHelper
{
    protected static function dateTimeFormat(): string
    {
        return 'Y-m-d\TH:i:s'; // example 2022-11-21T14:32:25
    }

    /**
     * @param  string  $dateTimeString string in the WooCommerce API date time format (example 2022-11-21T14:32:25)
     *               The passed string MUST be in UTC/GMT timezone or the alternative timezone must be provided
     * @param \DateTimeZone|null if $dateTimeString is not in UTC/GMT timezone then the correct timezone MUST be provided.
     *              The easiest way to use this is to only pass fields like date_created_gmt and avoid using date_created
     */
    public static function dateTimeStringToObject(
        string $dateTimeString,
        ?\DateTimeZone $dateTimeZone = null
    ): CarbonImmutable {
        if (! $dateTimeZone) {
            $dateTimeZone = new \DateTimeZone('UTC');
        }

        $dateTimeObject = CarbonImmutable::createFromFormat(
            self::dateTimeFormat(),
            $dateTimeString,
            $dateTimeZone
        );

        if ($dateTimeObject === false) {
            throw new DateTimeHelperException('Unable to parse provided WooCommerce datetime string');
        }

        return $dateTimeObject;
    }

    /**
     * @param  ?CarbonImmutable  $dateTimeObject This can be in any timezone it likes
     * @return ?string the returned strings timezone will be UTC/GMT
     */
    public static function dateTimeObjectToString(?CarbonImmutable $dateTimeObject): ?string
    {

        if (! $dateTimeObject) {
            return null;
        }

        return $dateTimeObject
            ->setTimezone(new \DateTimeZone('UTC'))
            ->format('Y-m-d\TH:i:sP');
        //            ->format(self::dateTimeFormat());
    }
}
