<?php

namespace Modules\WooCommerce\Jobs\Abstractions;

use App\Abstractions\Integrations\ClientResponseDataInterface;
use App\Models\Concerns\HasJobPayload;
use DateTime;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use InvalidArgumentException;
use Modules\WooCommerce\Data\WooCommerceResponseData;

/**
 * @method release(int $delay)
 *
 * @property $job
 */
abstract class WooCommerceJob implements ShouldQueue
{
    use Dispatchable;
    use HasJobPayload;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    public ?array $jobPayload = [];

    public int $uniqueFor = 15 * 60;

    public int $timeout = 15 * 60;

    public int $maxExceptions = 1;

    abstract public function apiCall(): WooCommerceResponseData|ClientResponseDataInterface;

    /**
     * Initialization steps for every job.
     *
     * @throws InvalidArgumentException
     */
    protected function jobStarted(): void
    {
        $job = $this->getJobPayload();
        if ($job) {
            $this->jobPayload = $job->payload;
        }
    }

    /**
     * @throws InvalidArgumentException
     */
    protected function jobCompleted(): void
    {
        $this->deleteJobPayload();
    }

    protected function nextPage(): void
    {
        // Release only works for redis queue driver, not for sync driver.
        $this->release(0);
    }

    /**
     * Set the value of timeout
     *
     * @return  self
     */
    public function setTimeout($timeout): static
    {
        $this->timeout = $timeout;

        $this->uniqueFor = $this->timeout;

        return $this;
    }

    /**
     * Determine the time at which the job should time out.
     */
    public function retryUntil(): DateTime
    {
        return now()->addSeconds($this->timeout);
    }

    /**
     * Execute the job.
     *
     * @throws Exception
     */
    public function handle(): void
    {
        //Get job payload from database.
        $this->jobStarted();

        //Make API call.
        $wooCommerceResponseDto = $this->apiCall();

        //Save payload in jobs
        $this->saveJobPayload([
            'nextPage' => $wooCommerceResponseDto->nextPage,
            'modifiedAfterDateTime' => $wooCommerceResponseDto->modifiedAfterDateTime,
        ]);

        //Get next page
        if ($wooCommerceResponseDto->nextPage) {
            $this->nextPage();
        } else {
            $this->jobCompleted();
        }
    }
}
