<?php

namespace Modules\WooCommerce\Entities;

use App\Abstractions\Integrations\ApiDataTransformerInterface;
use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Abstractions\Integrations\SalesChannels\AbstractSalesChannelProduct;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductManagerInterface;
use App\Abstractions\Integrations\SalesChannels\SalesChannelProductRepositoryInterface;
use App\DTO\ProductDto;
use App\DTO\ProductImageDto;
use App\Helpers;
use App\Importers\Parsers\FieldParserFactory;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use App\Models\Product;
use App\Models\ProductListing;
use App\Models\Setting;
use App\Repositories\DataImportMappingRepository;
use Carbon\CarbonImmutable;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Arr;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetProductsAdt;
use Modules\WooCommerce\Database\Factories\WooCommerceProductFactory;
use Modules\WooCommerce\Managers\WooCommerceProductManager;
use Modules\WooCommerce\Repositories\WooCommerceProductRepository;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Spatie\LaravelData\DataCollection;

/**
 * @property int $id
 * @property int $integration_instance_id
 * @property-read int $woo_commerce_id
 * @property-read string $sku
 * @property-read string $name
 * @property-read string $slug
 * @property-read string $date_created_gmt
 * @property-read string $date_modified_gmt
 * @property-read string $type
 * @property-read string $status
 * @property-read bool $featured
 * @property-read string $catalog_visibility
 * @property-read float $price
 * @property-read float $weight
 * @property-read float $dimensions_length
 * @property-read float $dimensions_width
 * @property-read float $dimensions_height
 * @property array $json_object
 * @property-read CarbonImmutable $created_at
 * @property ?CarbonImmutable $updated_at
 * @property ?CarbonImmutable $archived_at
 * @property-read WooCommerceIntegrationInstance $integrationInstance
 * @property-read ProductListing $productListing
 *
 * @method static dataFeedBulkImport(array $array)
 */
class WooCommerceProduct extends AbstractSalesChannelProduct
{
    use HasFactory;
    use HasFilters;
    use HasSort;

    public function __construct(array $attributes = [])
    {
        parent::__construct($attributes);
    }

    public static function getIntegrationName(): string
    {
        return 'woo-commerce';
    }

    public static function repository(): SalesChannelProductRepositoryInterface
    {
        return app(WooCommerceProductRepository::class);
    }

    public static function getUniqueField(): string
    {
        return 'woo_commerce_id';
    }

    public function isActive(): bool
    {
        return $this->status === 'publish';
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function integrationInstance(): BelongsTo
    {
        return $this->belongsTo(WooCommerceIntegrationInstance::class);
    }

    /*
    |--------------------------------------------------------------------------
    | Methods
    |--------------------------------------------------------------------------
    */

    public function delete(): ?bool
    {
        $this->productListing()->delete();

        return parent::delete();
    }

    public function getImages(): DataCollection
    {
        $images = collect($this->json_object['images'] ?? []);
        $primaryUrl = $images->pluck('src')->first();

        return ProductImageDto::collection($images->map(function ($image) use ($primaryUrl) {
            return ProductImageDto::from([
                'sku' => $this->applyDataMapping($this, 'sku'),
                'url' => $image['src'],
                'name' => $image['name'],
                'is_primary' => $image['src'] === $primaryUrl,
            ]);
        }));
    }

    // TODO: Default mapping for tags would make sense here
    /**
     * @throws NotFoundExceptionInterface
     * @throws ContainerExceptionInterface
     */
    public function getProductDto(): ProductDto
    {
        return ProductDTO::from([
            'sku' => $this->applyDataMapping($this, 'sku'),
            'name' => $this->applyDataMapping($this, 'item_name'),
            'type' => Product::TYPE_STANDARD,
            'weight' => (float) $this->applyDataMapping($this, 'weight'),
            'weight_unit' => Helpers::setting(Setting::KEY_DEFAULT_WEIGHT_UNIT),
            'length' => (float) $this->applyDataMapping($this, 'length'),
            'width' => (float) $this->applyDataMapping($this, 'width'),
            'height' => (float) $this->applyDataMapping($this, 'height'),
            'dimensions_unit' => Helpers::setting(Setting::KEY_DEFAULT_DIMENSION_UNIT),
            'images' => $this->getImages(),
            'pricing' => $this->getPricing(),
        ]);
    }



    // TODO: Review this... and maybe abstract it to AbstractSalesChannelProduct

    /**
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function applyDataMapping(mixed $data, string $mappedColumn): mixed
    {
        if (!$mappings = config()->get('woo_commerce_mappings')) {
            $mappings = app(DataImportMappingRepository::class)->getListingMappings($this->integration_instance_id);
        }

        $skuMapping = $mappings->where('sku_field', $mappedColumn)->first();
        $listingField = @$skuMapping['listing_field'];

        if (is_null($listingField)) {
            switch ($mappedColumn) {
                case 'item_name':
                    $listingField = 'name';
                    break;

                case 'sku':
                case 'sales_channel_listing_id':
                    $listingField = 'sku';
                    break;

                case 'price':
                    $listingField = 'price';
                    break;

                case 'weight':
                    $listingField = 'weight';
                    break;

                case 'length':
                    $listingField = 'dimensions_length';
                    break;

                case 'width':
                    $listingField = 'dimensions_width';
                    break;

                case 'height':
                    $listingField = 'dimensions_height';
                    break;
            }
        }

        //Return value
        $returnValue = @$data[$listingField];

        //Apply mappings
        if ($returnValue && @$skuMapping['parsers'] && is_array($skuMapping['parsers'])) {
            foreach ($skuMapping['parsers'] as $parser) {
                $returnValue = FieldParserFactory::make($parser['rule'], $parser['args'], $data)->parse($returnValue);
            }
        }

        return $returnValue;
    }

    public static function getSkuField(): string
    {
        return 'sku';
    }

    public static function getPriceField(): string
    {
        return 'price';
    }

    // TODO: add to abstraction and implement on other sales channels
    public static function getLastModified(): string
    {
        return 'date_modified_gmt';
    }

    public static function newFactory(): Factory
    {
        return WooCommerceProductFactory::new();
    }

    public function getListingUrl(): ?string
    {
        return 'https://'.$this->integrationInstance->integration_settings['url'].'/product/'.$this->slug;
    }

    /*
    |--------------------------------------------------------------------------
    | Implementers
    |--------------------------------------------------------------------------
    */

    // TODO: These should be static functions
    public function availableColumns(): array
    {
        $this->availableColumns = array_merge(Arr::except($this->getAllColumnsAndTypes(), [
            'integration_instance_id',
            'json_object',
        ]), [
            'product' => 'product'
        ]);
        return $this->availableColumns;
    }

    public function visibleColumns(): array
    {
        return array_keys(Arr::except($this->availableColumns, [
            'created_at',
            'updated_at',
        ]));
    }

    public static function specialLabels(): array
    {
        return [
            'sku' => 'SKU',
            'woo_commerce_id' => 'ID',
            'date_created_gmt' => 'Created Date',
            'date_modified_gmt' => 'Updated Date',
            'dimensions_length' => 'Length',
            'dimensions_width' => 'Width',
            'dimensions_height' => 'Height',
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['id', 'sku', 'name'];
    }

    public function manager(WooCommerceIntegrationInstance|IntegrationInstanceInterface $integrationInstance): SalesChannelProductManagerInterface
    {
        return new WooCommerceProductManager($integrationInstance);
    }

    public function refreshAdt(): ApiDataTransformerInterface
    {
        return new WooCommerceGetProductsAdt();
    }

    public function isUsed(): bool|array
    {
        $linkedOrders = WooCommerceOrderItem::where('sku', $this->sku)->count();

        if ($linkedOrders) {
            return [
                'woo-commerce.orders' => trans_choice('messages.currently_used', $linkedOrders, [
                    'resource' => 'WooCommerce Order',
                    'model' => 'WooCommerce Listing('.$this->sku.')',
                ]),
            ];
        }

        return false;
    }
}
