<?php

namespace Modules\WooCommerce\Console;

use Carbon\CarbonImmutable;
use DateInterval;
use Illuminate\Console\Command;
use Modules\WooCommerce\ApiDataTransferObjects\WooCommerceGetProductsAdt;
use Modules\WooCommerce\Entities\WooCommerceIntegrationInstance;
use Modules\WooCommerce\Jobs\RefreshWooCommerceProductsJob;
use Modules\WooCommerce\Repositories\WooCommerceIntegrationInstanceRepository;

class RefreshWooCommerceProductsCommand extends Command
{
    public function __construct(private readonly WooCommerceIntegrationInstanceRepository $integrationInstanceRepository)
    {
        parent::__construct();
    }

    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'woo-commerce:products:refresh {--lastUpdatedAfter=} {--interval=} {--integrationInstanceId=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Refresh Woocommerce Products';

    public function handle(): int
    {
        $lastUpdatedAfter = $this->determineLastUpdatedAfter();

        if ($this->option('integrationInstanceId')) {
            $query = $this->integrationInstanceRepository->all($this->option('integrationInstanceId'));
        } else {
            $query = $this->integrationInstanceRepository->all();
        }

        $this->info('Refreshing Woocommerce Products for '.$query->count().' integrationInstances');

        $query
            ->each(function (WooCommerceIntegrationInstance $wooCommerceIntegrationInstance) use ($lastUpdatedAfter) {
                RefreshWooCommerceProductsJob::dispatch(
                    $wooCommerceIntegrationInstance,
                    new WooCommerceGetProductsAdt(
                        $lastUpdatedAfter
                    )
                );
            });

        return 0;
    }

    protected function determineLastUpdatedAfter(): ?CarbonImmutable
    {
        $intervalString = $this->option('interval');
        $lastUpdatedAfterString = $this->option('lastUpdatedAfter');

        if ($intervalString && $lastUpdatedAfterString) {
            throw new \Exception('You cannot provide both --interval and --lastUpdatedAfter');
        }

        if ($intervalString) {
            $interval = DateInterval::createFromDateString($intervalString);

            return (new CarbonImmutable())->sub($interval);
        }

        if ($lastUpdatedAfterString) {
            return CarbonImmutable::parse($lastUpdatedAfterString);
        }

        return null;
    }
}
