<?php

namespace Modules\Veracore\Tests\Feature;

use App\Http\Requests\StoreInventoryAdjustment;
use App\Jobs\UncancellableVeracoreOrderJob;
use App\Models\CustomField;
use App\Models\CustomFieldValue;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Product;
use App\Models\SalesChannel;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\Warehouse;
use App\Repositories\SalesOrder\SalesOrderRepository;
use Illuminate\Database\Eloquent\HigherOrderBuilderProxy;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Queue;
use Illuminate\Support\HigherOrderCollectionProxy;
use Illuminate\Testing\TestResponse;
use Modules\Veracore\Entities\VeracoreIntegrationInstance;
use Modules\Veracore\Entities\VeracoreOrder;
use Modules\Veracore\Services\Fulfillments\VeracoreDispatcher;
use Modules\Veracore\Services\VeracoreClient;
use Modules\Veracore\Services\VeracoreManager;
use Modules\Veracore\Tests\VeracoreTest;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;

class VeracoreOrdersTest extends VeracoreTest
{

    use FastRefreshDatabase;
    use WithFaker;


    /**
     * @param  TestResponse  $response
     * @return HigherOrderBuilderProxy|Model|HigherOrderCollectionProxy|mixed|object|null
     */
    public function getFulfillment(TestResponse $response): mixed
    {
        $fulfillment = SalesOrder::query()->find($response->json('data.id'))
            ->salesOrderFulfillments->first();

        $orderId = VeracoreOrder::query()->where('sku_fulfillment_id', $fulfillment->id)->first()->veracore_id;

        Http::fake([
            VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Body>
                                <CancelOrderResponse xmlns="http://omscom/">
                                    <CancelOrderResult>
                                        <OrderSeqID>123</OrderSeqID>
                                        <OrderID>'.$orderId.'</OrderID>
                                    </CancelOrderResult>
                                </CancelOrderResponse>
                            </soap:Body>
                        </soap:Envelope>'
            )
        ]);

        $this->assertDatabaseHas('sales_order_fulfillments', [
            'id' => $fulfillment->id,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
            'sales_order_id' => $response->json('data.id')
        ]);
        return $fulfillment;
    }

    protected function setUp(): void
    {
        parent::setUp();
        $this->fakeAccessTokenRequest();
    }


    protected function createFulfillmentOrderWithoutDeliverByDate(int $status = 200): TestResponse
    {

        $product = Product::factory()->create();
        $warehouse = Warehouse::query()->firstOrCreate(['type' => Warehouse::TYPE_3PL])->withDefaultLocation();
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'adjustment_date' => now(),
            'quantity' => 3,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10
        ])->assertSuccessful();
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'sales_channel_id' => SalesChannel::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'description' => 'test-product',
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE
        ])->assertStatus($status);

        return $response;
    }


    public function test_it_records_api_log_when_veracore_fulfillment_fails(){

        Queue::fake();

        /** @var VeracoreIntegrationInstance $instance */
        $instance = VeracoreIntegrationInstance::query()->firstOrFail();
        $instance->disableDeliverByDateCheck();

        Http::fake([
            'rhu003.veracore.com/pmomsws/oms.asmx' => Http::response(
                'Internal server error',
                500
            ),
        ]);

        $this->fakeGetOrderInfo();

        $this->assertDatabaseEmpty('api_logs');
        $this->createFulfillmentOrderWithoutDeliverByDate(400);

        // Api log should be created
        Queue::assertClosurePushed();
    }

    public function test_it_will_submit_orders_without_delivery_date_to_veracore_when_setting_is_not_enabled(){

        Queue::fake();

        /** @var VeracoreIntegrationInstance $instance */
        $instance = VeracoreIntegrationInstance::query()->firstOrFail();
        $instance->disableDeliverByDateCheck();

        $this->fakeAddOrderRequest();
        $this->fakeGetOrderInfo();

        $this->createFulfillmentOrderWithoutDeliverByDate();

        $this->assertDatabaseCount('veracore_orders', 1);
        Queue::assertClosurePushed();

    }


    public function test_it_wont_submit_orders_without_delivery_date_to_veracore_when_setting_is_enabled(){

        /** @var VeracoreIntegrationInstance $instance */
        $instance = VeracoreIntegrationInstance::query()->firstOrFail();
        $instance->enforceDeliverByDate();

        $this->createFulfillmentOrderWithoutDeliverByDate(400);

        Http::assertNothingSent();
        $this->assertDatabaseEmpty('api_logs');

    }

    public function test_it_can_get_veracore_tracking_info(){

        Queue::fake();

        // Prepare orders
        $id1 = '53667';
        $id2 = '53668';

        $integrationInstance = IntegrationInstance::factory()->create([
            'name' => Integration::NAME_SHOPIFY,
            'integration_id' => Integration::query()->firstOrCreate(
                ['name' => Integration::NAME_SHOPIFY],
                ['integration_type' => Integration::TYPE_SALES_CHANNEL]
            )->id,
            'is_automatic_sync_enabled' => true,
        ]);
        /** @var SalesChannel $salesChannel */
        $salesChannel = SalesChannel::factory()->create([
            'integration_instance_id' => $integrationInstance->id
        ]);

        $salesOrder = SalesOrder::factory()->create([
            'sales_channel_id' => $salesChannel->id,
        ]);

        $fulfillments = SalesOrderFulfillment::factory(2)->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
            'sales_order_id' => $salesOrder->id
        ]);

        VeracoreOrder::factory()->create(
            [
                'json_data' => [
                    'Order' => [
                        'OrderID' => $id1
                    ]
                ],
                'sku_fulfillment_id' => $fulfillments->first->id
            ]
        );
        VeracoreOrder::factory()->create(
            [
                'json_data' => [
                    'Order' => [
                        'OrderID' => $id2
                    ]
                ],
                'sku_fulfillment_id' => $fulfillments->last()->id
            ]
        );

        $tracking1 = $this->faker->unique()->md5();
        $tracking2 = $this->faker->unique()->md5();

        // Mock endpoints
        Http::fake([
            VeracoreClient::GET_PACKAGES_ENDPOINT . '?*' => Http::response([
                'ShippingUnits' => [
                    [
                        'Order' => [
                            'OrderID' => $id1,
                            'ReferenceNumber' => $this->faker->unique()->word(),
                            'CurrentOrderStatus' => 'Fulfilled',
                        ],
                        'Shipping' => [
                            'UTCShippedDateTime' => now()->subDay(),
                            'FreightCarrier' => 'UPS',
                            'FreightService' => 'STANDARD',
                            'FreightCodeDescription' => 'Ground',
                            'Freight' => [
                                'MarkedUpFreight' => $this->faker->numberBetween(3, 8)
                            ],
                            'TrackingNumber' => $tracking1,
                            'ShipTo' => [
                                'Company' => "Test\nCompany",
                            ]
                        ]
                    ],
                    [
                        'Order' => [
                            'OrderID' => $id2,
                            'ReferenceNumber' => $this->faker->unique()->word(),
                            'CurrentOrderStatus' => 'Fulfilled',
                        ],
                        'Shipping' => [
                            'UTCShippedDateTime' => now()->subDay(),
                            'FreightCarrier' => 'UPS',
                            'FreightService' => 'STANDARD',
                            'FreightCodeDescription' => 'Ground',
                            'Freight' => [
                                'MarkedUpFreight' => $this->faker->numberBetween(3, 8)
                            ],
                            'TrackingNumber' => $tracking2
                        ]
                    ]
                ]
            ])
        ]);


        // Download orders
        /** @var VeracoreManager $manager */
        $manager = app(VeracoreManager::class);
        $manager->updateTrackingInfo();

        // Assert
        $this->assertDatabaseCount('veracore_orders', 2);
        $this->assertDatabaseHas('veracore_orders', [
            'veracore_id' => $id1
        ]);
        $this->assertDatabaseHas('veracore_orders', [
            'veracore_id' => $id2
        ]);

        // Tracking should be updated
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'id' => $fulfillments->first()->id,
            'tracking_number' => $tracking1,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED
        ]);
        $this->assertDatabaseHas('sales_order_fulfillments', [
            'id' => $fulfillments->last()->id,
            'tracking_number' => $tracking2,
            'status' => SalesOrderFulfillment::STATUS_FULFILLED
        ]);

        // Sales order should be fulfilled and closed.
        $this->assertDatabaseHas('sales_orders', [
            'id' => $salesOrder->id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_FULFILLED,
            'order_status' => SalesOrder::STATUS_CLOSED
        ]);

        Queue::assertClosurePushed();

    }

    public function test_it_can_dispatch_fulfillment_to_veracore(){

        Queue::fake();

        $giftCard = CustomField::factory()->create([
            'name' => 'Gift Card Note',
            'link_type' => 'SalesOrder',
        ]);

        $veracore = VeracoreIntegrationInstance::active();
        $settings = $veracore->integration_settings;
        $settings[IntegrationInstance::GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD] = $giftCard->id;
        $veracore->update(['integration_settings' => $settings]);

        // Prepare
        $product = Product::factory()->create();
        $warehouse = Warehouse::query()->firstOrCreate(['type' => Warehouse::TYPE_3PL])->withDefaultLocation();
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $product->id,
            'adjustment_date' => now(),
            'quantity' => 3,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10
        ])->assertSuccessful();
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'deliver_by_date' => now()->addDays(2),
            'sales_order_lines' => [
                [
                    'product_id' => $product->id,
                    'quantity' => 3,
                    'description' => 'test-product',
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        CustomFieldValue::factory()->create([
            'link_type' => SalesOrder::class,
            'link_id' => $response->json('data.id'),
            'custom_field_id' => $giftCard->id,
        ]);

        $orders = new SalesOrderRepository;
        $fulfillment = $orders->createFulfillment(
            order: $orders->findById($response->json('data.id')),
            payload: [
                'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
                'fulfilled_at' => now(),
                'warehouse_id' => $warehouse->id,
                'fulfillment_lines' => [
                    [
                        'sales_order_line_id' => $response->json('data.item_info.0.sales_order_line_id'),
                        'quantity' => 3
                    ]
                ]
            ]
        );

        $this->fakeGetOrderInfo();
        $this->fakeAddOrderRequest();

        // Action
        $dispatcher = app(VeracoreDispatcher::class);
        $dispatcher->dispatchFulfillmentToProvider($fulfillment);

        // Assertions
        $this->assertDatabaseHas('sales_orders', [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
        ]);

        $this->assertDatabaseHas('sales_order_fulfillments', [
            'id' => $fulfillment->id,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED
        ]);

        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => $fulfillment->id,
            'ReferenceNumber' => VeracoreManager::VERACORE_TEST_REFERENCE,
        ]);

        Queue::assertClosurePushed();
    }

    public function test_it_cancels_veracore_order_when_sku_fulfillment_is_deleted(): void{

        Queue::fake();

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
        ]);

        $orderId = $this->faker->unique()->numberBetween(1, 1000);
        VeracoreOrder::factory()->create([
            'sku_fulfillment_id' => $fulfillment->id,
            'json_data' => [
                'Order' => [
                    'OrderID' => $orderId
                ]
            ]
        ]);

        Http::fake([
            VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Body>
                                <CancelOrderResponse xmlns="http://omscom/">
                                    <CancelOrderResult>
                                        <OrderSeqID>123</OrderSeqID>
                                        <OrderID>'.$orderId.'</OrderID>
                                    </CancelOrderResult>
                                </CancelOrderResponse>
                            </soap:Body>
                        </soap:Envelope>'
            )
        ]);

        $this->fakeGetOrderInfo();

        $this->deleteJson('/api/sales-order-fulfillments/'.$fulfillment->id)->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => null,
        ]);
        Queue::assertClosurePushed();

    }


    /**
     * @return void
     */
    public function test_it_indicates_in_api_logs_when_veracore_order_cancellation_fails(): void{

        Queue::fake();

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
        ]);

        $orderId = $this->faker->unique()->numberBetween(1, 1000);
        VeracoreOrder::factory()->create([
            'sku_fulfillment_id' => $fulfillment->id,
            'json_data' => [
                'Order' => [
                    'OrderID' => $orderId
                ]
            ]
        ]);

        Http::fake([
            VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                            <s:Envelope xmlns:s="http://schemas.xmlsoap.org/soap/envelope/">
                                <s:Body>
                                    <CancelOrderResponse xmlns="http://omscom/">
                                        <CancelOrderResult xmlns:a="http://schemas.datacontract.org/2004/07/ProMail.WebServices.Objects.OrderAdjustments" xmlns:i="http://www.w3.org/2001/XMLSchema-instance">
                                            <a:ExceptionMessage>You have to provide an order ID to process the request!</a:ExceptionMessage>
                                            <a:OrderID i:nil="true"/>
                                            <a:OrderSeqID>0</a:OrderSeqID>
                                        </CancelOrderResult>
                                    </CancelOrderResponse>
                                </s:Body>
                            </s:Envelope>'
            )
        ]);

        $this->fakeGetOrderInfo();

        $this->deleteJson('/api/sales-order-fulfillments/'.$fulfillment->id)->assertServerError();

        // Fulfillment shouldn't be deleted.
        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => $fulfillment->id,
        ]);
        Queue::assertClosurePushed();

    }


    public function test_it_wont_delete_fulfillment_if_veracore_order_cannot_be_canceled(): void{

        Queue::fake();

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
        ]);

        $orderId = $this->faker->unique()->numberBetween(1, 1000);
        VeracoreOrder::factory()->create([
            'sku_fulfillment_id' => $fulfillment->id,
            'json_data' => [
                'Order' => [
                    'OrderID' => $orderId
                ]
            ]
        ]);



        Http::fake([
            VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT => Http::response(
                'Bad Request',
                400
            )
        ]);

        Http::fake([
            VeracoreClient::ACCESS_TOKEN_ENDPOINT => Http::response([
                'UtcExpirationDate' => now()->addWeek(),
                'Token' => $this->faker->sha1(),
            ])
        ]);

        Http::fake([
            VeracoreClient::BASE_ENDPOINT . '?op=GetOrderInfo' => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                    <soap:Envelope
                        xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"
                        xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                        xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                        <soap:Header>
                            <DebugHeader
                                xmlns="http://omscom/">
                                <Debug>false</Debug>
                            </DebugHeader>
                        </soap:Header>
                        <soap:Body>
                            <GetOrderInfoResponse
                                xmlns="http://omscom/">
                                <GetOrderInfoResult>
                                    <OrdHead>
                                        <Status>
                                            <Shipped>true</Shipped>
                                        </Status>
                                    </OrdHead>
                                    <OfferInfo>
                                        <OfferType>
                                            <OfferId>LB-MT-CK-FZ-2LB-1</OfferId>
                                            <OfferDesc>1 Unit - Large Lobster Meat 2lb</OfferDesc>
                                            <OrderQty>1</OrderQty>
                                            <UnitPrice>0</UnitPrice>
                                        </OfferType>
                                    </OfferInfo>
                                </GetOrderInfoResult>
                            </GetOrderInfoResponse>
                        </soap:Body>
                    </soap:Envelope>'
            )
        ]);


        $this->deleteJson('/api/sales-order-fulfillments/'.$fulfillment->id)
            ->assertServerError();

        $this->assertDatabaseCount('sales_order_fulfillments', 1);
        $this->assertDatabaseCount('veracore_orders', 1);
        Queue::assertClosurePushed();

        Queue::assertPushed(UncancellableVeracoreOrderJob::class);
    }

    public function test_it_cancels_veracore_order_when_sales_order_shipping_address_changes(): void{

        Queue::fake();

        $fulfillment = SalesOrderFulfillment::factory()->create([
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE,
            'status' => SalesOrderFulfillment::STATUS_SUBMITTED,
        ]);

        $orderId = $this->faker->unique()->numberBetween(1, 1000);
        VeracoreOrder::factory()->create([
            'sku_fulfillment_id' => $fulfillment->id,
            'json_data' => [
                'Order' => [
                    'OrderID' => $orderId
                ]
            ]
        ]);

        Http::fake([
            VeracoreClient::ORDER_ADJUSTMENTS_ENDPOINT => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Body>
                                <CancelOrderResponse xmlns="http://omscom/">
                                    <CancelOrderResult>
                                        <OrderSeqID>123</OrderSeqID>
                                        <OrderID>'.$orderId.'</OrderID>
                                    </CancelOrderResult>
                                </CancelOrderResponse>
                            </soap:Body>
                        </soap:Envelope>'
            )
        ]);

        $this->fakeGetOrderInfo();

        $this->putJson('/api/sales-orders/'.$fulfillment->sales_order_id, [
            'shipping_address' => [
                'first_name' => $this->faker->firstName,
                'last_name' => $this->faker->lastName,
                'address1' => $this->faker->streetAddress,
                'address2' => $this->faker->secondaryAddress,
                'city' => $this->faker->city,
                'state' => $this->faker->stateAbbr,
                'zip' => $this->faker->postcode,
                'country' => $this->faker->countryCode,
                'phone' => $this->faker->phoneNumber,
                'email' => $this->faker->email,
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => null,
        ]);
        Queue::assertClosurePushed();
    }


    public function test_it_cancels_veracore_order_when_sales_order_line_items_change(): void{

        Queue::fake();

        $this->fakeGetOrderInfo();
        $this->fakeAddOrderRequest();
        $this->fakeTrackingInfoRequest();

        /** @var VeracoreIntegrationInstance $instance */
        $instance = VeracoreIntegrationInstance::query()->firstOrFail();
        $instance->disableDeliverByDateCheck();

        $response = $this->createFulfillmentOrderWithoutDeliverByDate();

        Queue::assertClosurePushed();

        $this->assertDatabaseHas('sales_orders', [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_AWAITING_TRACKING,
        ]);

        $fulfillment = $this->getFulfillment($response);

        $this->putJson('/api/sales-orders/'.$fulfillment->sales_order_id, [
            'sales_order_lines' => [
                [
                    'id' => $response->json('data.item_info.0.sales_order_line_id'),
                    'quantity' => $response->json('data.item_info.0.item_quantity'),
                    'amount' => $response->json('data.item_info.0.item_price'),
                    'description' => $this->faker->sentence(),
                ],
                [ // Add new line
                    'product_id' => Product::factory()->create()->id,
                    'quantity' => 3,
                    'description' => 'test-product',
                    'amount' => 5,
                    'warehouse_id' => Warehouse::query()->firstOrCreate(['type' => Warehouse::TYPE_3PL])->id,
                ]
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => null,
        ]);
        Queue::assertClosurePushed();
        $this->assertDatabaseHas('sales_orders', [
            'id' => $fulfillment->sales_order_id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC
        ]);

    }

    public function test_it_cancels_veracore_order_when_bundle_line_quantity_changes(): void{

        Queue::fake();

        $this->fakeGetOrderInfo();
        $this->fakeAddOrderRequest();
        $this->fakeTrackingInfoRequest();

        /** @var VeracoreIntegrationInstance $instance */
        $instance = VeracoreIntegrationInstance::query()->firstOrFail();
        $instance->disableDeliverByDateCheck();

        /** @var Product $bundle */
        $bundle = Product::factory()->create(['type' => Product::TYPE_BUNDLE]);
        $component1 = Product::factory()->create();
        $component2 = Product::factory()->create();
        $bundle->setBundleComponents([
            ['id' => $component1->id, 'quantity' => 2],
            ['id' => $component2->id, 'quantity' => 3]
        ]);

        $warehouse = Warehouse::query()->firstOrCreate(['type' => Warehouse::TYPE_3PL])->withDefaultLocation();
        // Create stock via adjustments
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $component1->id,
            'adjustment_date' => now(),
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10
        ])->assertSuccessful();
        $this->postJson('/api/inventory-adjustments', [
            'product_id' => $component2->id,
            'adjustment_date' => now(),
            'quantity' => 10,
            'adjustment_type' => StoreInventoryAdjustment::ADJUSTMENT_TYPE_INCREASE,
            'warehouse_id' => $warehouse->id,
            'unit_cost' => 10
        ])->assertSuccessful();

        // Create sales order
        $response = $this->postJson('/api/sales-orders', [
            'currency_code' => 'USD',
            'order_date' => now(),
            'order_status' => SalesOrder::STATUS_OPEN,
            'deliver_by_date' => now()->addDays(2),
            'sales_channel_id' => SalesChannel::factory()->create()->id,
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'sales_channel_line_id' => 'test',
                    'quantity' => 1,
                    'description' => 'test-product',
                    'amount' => 5,
                    'warehouse_id' => $warehouse->id,
                ]
            ]
        ])->assertSuccessful();

        $this->postJson('/api/sales-orders/'.$response->json('data.id').'/fulfill', [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_VERACORE
        ])->assertSuccessful();


        $fulfillment = $this->getFulfillment($response);

        // Increase bundle quantity
        $this->putJson('/api/sales-orders/'.$fulfillment->sales_order_id, [
            'sales_order_lines' => [
                [
                    'product_id' => $bundle->id,
                    'sales_channel_line_id' => 'test',
                    'quantity' => 3,
                    'amount' => $response->json('data.item_info.0.item_price'),
                    'description' => 'update bundle product',
                ]
            ]
        ])->assertSuccessful();

        $this->assertDatabaseCount('sales_order_fulfillments', 0);
        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseHas('veracore_orders', [
            'sku_fulfillment_id' => null,
        ]);
        Queue::assertClosurePushed();
        $this->assertDatabaseHas('sales_orders', [
            'id' => $fulfillment->sales_order_id,
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_OUT_OF_SYNC
        ]);
    }


    public function test_it_submits_orders_to_veracore_when_delivery_date_changes(){

        $this->markTestSkipped('Needs to review auto fulfillment when deliver by date changes.');
        $this->fakeAddOrderRequest();

        $response = $this->createFulfillmentOrderWithoutDeliverByDate(400);

        $this->assertDatabaseHas('sales_orders', [
            'id' => $response->json('data.id'),
            'fulfillment_status' => SalesOrder::FULFILLMENT_STATUS_UNFULFILLED,
        ]);

        // No submission or record should be available for veracore
        // since there's no receive by date.
        $this->assertDatabaseEmpty('veracore_orders');

        // Update receive by date
        $this->putJson('/api/sales-orders/' . $response->json('data.id'), [
            'deliver_by_date' => now()->addDays(2)
        ])->assertSuccessful();

        $this->assertDatabaseCount('veracore_orders', 1);
        $this->assertDatabaseCount('api_logs', 1);

    }


    protected function fakeAccessTokenRequest(): void{
        Http::fake([
            VeracoreClient::ACCESS_TOKEN_ENDPOINT => Http::response([
                'UtcExpirationDate' => now()->addWeek(),
                'Token' => $this->faker->sha1(),
            ])
        ]);
    }

    protected function fakeGetOrderInfo(?int $orderId = null, ?string $reference = null): void
    {
        $this->fakeAccessTokenRequest();

        $reference = $reference ?: null;
        $orderId = $orderId ?: null;

        Http::fake([
            VeracoreClient::BASE_ENDPOINT . '?op=GetOrderInfo' => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                                <soap:Envelope
                                    xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/"
                                    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
                                    xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                                    <soap:Header>
                                        <DebugHeader
                                            xmlns="http://omscom/">
                                            <Debug>false</Debug>
                                        </DebugHeader>
                                    </soap:Header>
                                    <soap:Body>
                                        <GetOrderInfoResponse
                                            xmlns="http://omscom/">
                                            <GetOrderInfoResult>
                                                <OrdHead>
                                                    <OrderId>'.$orderId.'</OrderId>
                                                    <Status>
                                                        <Picked>false</Picked>
                                                        <Shipped>false</Shipped>
                                                        <Complete>false</Complete>
                                                    </Status>
                                                    <ReferenceNo>'.$reference.'</ReferenceNo>
                                                </OrdHead>
                                                <OfferInfo>
                                                    <OfferType>
                                                        <OfferId>LB-MT-CK-FZ-2LB-1</OfferId>
                                                        <OfferDesc>1 Unit - Large Lobster Meat 2lb</OfferDesc>
                                                        <OrderQty>1</OrderQty>
                                                        <UnitPrice>0</UnitPrice>
                                                    </OfferType>
                                                </OfferInfo>
                                            </GetOrderInfoResult>
                                        </GetOrderInfoResponse>
                                    </soap:Body>
                                </soap:Envelope>'
            )
        ]);
    }

    protected function fakeTrackingInfoRequest(?int $id = null, ?string $trackingNumber = null): void{
        // This is used to fake
        Http::fake([
            VeracoreClient::makeUrl(VeracoreClient::GET_PACKAGES_ENDPOINT) . '?*' => Http::response([
                'ShippingUnits' => [
                    [
                        'Order' => [
                            'OrderID' => $id ?: -1, // Default to ID that doesn't exist so the order isn't found.
                            'ReferenceNumber' => VeracoreManager::VERACORE_TEST_REFERENCE,
                            'CurrentOrderStatus' => 'Fulfilled',
                        ],
                        'Shipping' => [
                            'UTCShippedDateTime' => now()->subDay(),
                            'FreightCarrier' => 'UPS',
                            'FreightService' => 'STANDARD',
                            'FreightCodeDescription' => 'Ground',
                            'Freight' => [
                                'MarkedUpFreight' => $this->faker->numberBetween(3, 8)
                            ],
                            'TrackingNumber' => $trackingNumber ?: $this->faker->md5(),
                            'ShipTo' => [
                                'Company' => "Test\nCompany",
                            ]
                        ]
                    ],
                ]
            ])
        ]);
    }

    protected function fakeAddOrderRequest(): void
    {
        Http::fake([
            'rhu003.veracore.com/pmomsws/oms.asmx' => Http::response(
                '<?xml version="1.0" encoding="utf-8"?>
                        <soap:Envelope xmlns:soap="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
                            <soap:Body>
                                <AddOrderResponse xmlns="http://omscom/">
                                    <AddOrderResult>
                                        <OrderSeqID>183</OrderSeqID>
                                        <OrderID>3</OrderID>
                                    </AddOrderResult>
                                </AddOrderResponse>
                            </soap:Body>
                        </soap:Envelope>'
            )
        ]);
    }
}
