<?php

namespace Modules\Veracore\Services;

use App\Exceptions\IntegrationInstance\UncancellableShippingProviderOrderException;
use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Jobs\UncancellableVeracoreOrderJob;
use App\Models\IntegrationInstance;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Services\ShippingProvider\ShippingProviderOrder;
use App\Services\ShippingProvider\ShippingProviderOrderManager;
use App\Utility\PhoneNumberUtility;
use Carbon\Carbon;
use Exception;
use Illuminate\Database\Eloquent\Model;
use Modules\Veracore\Entities\VeracoreIntegrationInstance;
use Modules\Veracore\Exceptions\VeracoreOrderDoesNotExistException;
use Modules\Veracore\Repositories\VeracoreOrderRepository;
use Throwable;

class VeracoreManager extends ShippingProviderOrderManager
{
    const VERACORE_TEST_REFERENCE = 'TEST';


    /**
     * @var VeracoreIntegrationInstance|Model
     */
    private VeracoreIntegrationInstance|Model $veracoreInstance;

    /**
     * @param  VeracoreOrderRepository  $orders
     */
    public function __construct(private readonly VeracoreOrderRepository $orders)
    {
        $this->veracoreInstance = VeracoreIntegrationInstance::active();
        $client = new VeracoreClient(
            credential: new CredentialManager($this->veracoreInstance),
        );
        parent::__construct(
            instance: $this->veracoreInstance,
            client: $client,
            orders: $this->orders
        );
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return ShippingProviderOrder
     * @throws SalesOrderFulfillmentDispatchException
     */
    public function createFulfillment(SalesOrderFulfillment $fulfillment): ShippingProviderOrder
    {
        $order = parent::createFulfillment($fulfillment);
        // Save the reference number for the order
        $this->orders->setReferenceNumber($order, $this->getOrderReference($fulfillment));
        return $order;
    }

    /**
     * @return bool
     */
    public function shouldOrdersHaveDeliveryByDate(): bool{
        return $this->veracoreInstance->shouldOrdersHaveDeliveryByDate();
    }


    /**
     * @param SalesOrderFulfillment $fulfillment
     * @return array
     */
    public function makeFulfillmentOrderPayload(SalesOrderFulfillment $fulfillment): array
    {

        $orderLines = $fulfillment->salesOrderFulfillmentLines->map(function (SalesOrderFulfillmentLine $line) {
            return [
                'sku'    => $line->salesOrderLine->product?->sku,
                'quantity'   => $line->quantity,
                'unit_price' => $line->salesOrderLine->amount
            ];
        });

        return [
            'id'                       => $fulfillment->id,
            'reference'                => $this->getOrderReference($fulfillment),
            'comments'                 => $this->getComments($fulfillment),
            'orderDate'                => $fulfillment->salesOrder->order_date,
            'orderLines'               => $orderLines,
            'customerFirstName'        => $fulfillment->salesOrder->shippingAddress?->name,
            'streetAddress1'           => $fulfillment->salesOrder->shippingAddress?->address1,
            'streetAddress2'           => $fulfillment->salesOrder->shippingAddress?->address2,
            'streetAddress3'           => $fulfillment->salesOrder->shippingAddress?->address3,
            'city'                     => $fulfillment->salesOrder->shippingAddress?->city,
            'state'                    => $fulfillment->salesOrder->shippingAddress?->province_code,
            'postalCode'               => $fulfillment->salesOrder->shippingAddress?->zip,
            'country'                  => $fulfillment->salesOrder->shippingAddress?->country,
            'neededBy'                 => Carbon::parse($fulfillment->salesOrder->deliver_by_date)->format('m/d/Y'),
            'releaseDate'              => Carbon::parse($fulfillment->salesOrder->ship_by_date)->toIso8601String(),
            'phone'                    => PhoneNumberUtility::applyMany(
                $fulfillment->salesOrder->shippingAddress?->phone,
                ['digitsOnly', 'withoutLeadingOne']
            ),
            'email'                    => $fulfillment->salesOrder->customer?->email,
            'fax'                      => $fulfillment->salesOrder->customer?->fax,
            'requestedShippingMethod'  => $fulfillment->requested_shipping_method,
        ];
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return string
     */
    private function getOrderReference(SalesOrderFulfillment $fulfillment): string
    {
        if(app()->environment() === 'production'){
            return $fulfillment->fulfillment_number;
        } else {
            return self::VERACORE_TEST_REFERENCE;
        }
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return string|null
     */
    private function getComments(SalesOrderFulfillment $fulfillment): ?string
    {
        if(@$this->veracoreInstance->integration_settings[IntegrationInstance::GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD]){
            return $fulfillment->salesOrder->customFieldValues
                ->where("custom_field_id", $this->veracoreInstance->integration_settings[IntegrationInstance::GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD])
                ->first()?->value;
        }
        return null;
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return void
     * @throws Throwable
     */
    public function cancelOrderForFulfillment(SalesOrderFulfillment $fulfillment): void
    {
        $order = $this->orders->getOrderByFulfillmentId($fulfillment->id);
        if($order){
            try {
                $currentOrder = $this->client->getOrder($order->veracore_id);
                if (!$this->isAlreadyCanceled($currentOrder)) {
                    if(!$this->canCancelVeracoreOrder($currentOrder)){
                        if(!$order->notified_about_cancellation){
                            dispatch((new UncancellableVeracoreOrderJob($fulfillment, $currentOrder, $order, auth()->id())))->onQueue('fulfillments');
                        }
                        throw new UncancellableShippingProviderOrderException('Order is either already shipped/completed or is in a state that cannot be canceled.');
                    }
                    $this->client->cancelOrder($order->veracore_id);
                }
            } catch (VeracoreOrderDoesNotExistException){
            }
            // We only unlink the sku fulfillment.
            // @see comments on SKU-6158
            $order->update(['sku_fulfillment_id' => null]);
        }
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return void
     * @throws SalesOrderFulfillmentDispatchException
     * @throws Throwable
     */
    public function updateOrderFromFulfillment(SalesOrderFulfillment $fulfillment): void
    {
        $order = $this->orders->getOrderByFulfillmentId($fulfillment->id);
        if(!$order){
            throw new SalesOrderFulfillmentDispatchException($fulfillment,'Order not found.');
        }

        // Veracore handles different updates differently.
        // There are different endpoints for adding new lines to the fulfillment,
        // and updating existing line quantities.

        $currentOrder = $this->client->getOrder($order->veracore_id);

        if(!$this->canUpdateVeracoreOrder($currentOrder)){
            throw new SalesOrderFulfillmentDispatchException($fulfillment,'Only unprocessed and incomplete veracore orders can be updated.');
        }

        $fulfillment->loadMissing('salesOrderFulfillmentLines.salesOrderLine.product');
        $addedLines = $this->getAddedFulfillmentLines($fulfillment, $currentOrder);

        if(!empty($addedLines)){
            $this->client->addLinesToFulfillmentOrder(
                $order->veracore_id,
                $addedLines
            );
        }

        $changingQuantities = $this->getFulfillmentLinesWithChangedQuantities($fulfillment, $currentOrder);

        if(!empty($changingQuantities)){
            $this->client->updateFulfillmentLineQuantities($order->veracore_id, $changingQuantities);
        }

    }

    private function canUpdateVeracoreOrder(array $order): bool
    {
        return ($order['OrdHead']['Status']['Unprocessed'] ?? '') == 'true' &&
            ($order['OrdHead']['Status']['Canceled'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Complete'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Picked'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Shipped'] ?? '') != 'true';
    }

    private function isAlreadyCanceled(array $order): bool
    {
        return ($order['OrdHead']['Status']['Canceled'] ?? '') == 'true';
    }

    private function canCancelVeracoreOrder(array $order): bool
    {
        return ($order['OrdHead']['Status']['Shipped'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Complete'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Pending'] ?? '') != 'true' &&
            ($order['OrdHead']['Status']['Picked'] ?? '') != 'true';
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @param  array  $veracoreOrder
     * @return array
     */
    private function getAddedFulfillmentLines(
        SalesOrderFulfillment $fulfillment,
        array $veracoreOrder
    ): array{

        return $fulfillment->salesOrderFulfillmentLines
            ->whereNotIn(
                'salesOrderLine.product.sku',
                collect(data_get($veracoreOrder, 'OfferInfo.OfferType'))->pluck('OfferId')->toArray()
            )->map(function(SalesOrderFulfillmentLine $fulfillmentLine){
                return [
                    'sku' => $fulfillmentLine->salesOrderLine->product->sku,
                    'quantity' => $fulfillmentLine->quantity,
                    'unit_price' => $fulfillmentLine->salesOrderLine->amount,
                    'description' => $fulfillmentLine->salesOrderLine->description,
                ];
            })->toArray();
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @param  array  $veracoreOrder
     * @return array
     */
    private function getFulfillmentLinesWithChangedQuantities(
        SalesOrderFulfillment $fulfillment,
        array $veracoreOrder
    ): array{

            $veracoreLines = collect(data_get($veracoreOrder, 'OfferInfo.OfferType'));

            return $fulfillment->salesOrderFulfillmentLines
                ->filter(function(SalesOrderFulfillmentLine $fulfillmentLine) use ($veracoreLines){
                    $veracoreLine = $veracoreLines->firstWhere('OfferId', $fulfillmentLine->salesOrderLine->product->sku);
                    return $veracoreLine && (int)$veracoreLine['OrderQty'] != $fulfillmentLine->quantity;
                })->map(function(SalesOrderFulfillmentLine $fulfillmentLine) use ($veracoreLines){
                    $veracoreLine = $veracoreLines->firstWhere('OfferId', $fulfillmentLine->salesOrderLine->product->sku);
                    return [
                        'sku' => $veracoreLine['OfferId'],
                        'quantity_difference' => $fulfillmentLine->quantity - (int)($veracoreLine['OrderQty'] ?? 0)
                    ];
                })->toArray();
    }

}