<?php

namespace Modules\Veracore\Entities;

use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\Warehouse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * @property Warehouse $linkedWarehouse
 */
class VeracoreIntegrationInstance extends IntegrationInstance
{
    use HasFactory;

    const IS_DELIVER_BY_DATE_REQUIRED = 'is_deliver_by_date_required';
    const AUTOMATE_FULFILLMENT = 'automate_fulfillment';
    protected $table = 'integration_instances';

    /**
     * @return IntegrationInstance|Model
     */
    public static function active(): VeracoreIntegrationInstance|Model{
        return static::query()->firstOrFail();
    }

    public static function exists(): bool{
        return static::query()->count() > 0;
    }

    /**
     * @return Attribute
     */
    public function linkedWarehouse(): Attribute{
        return Attribute::get(fn() => Warehouse::query()->findOrFail($this->integration_settings['linked_warehouse_id']));
    }

    /**
     * @return $this
     */
    public function enforceDeliverByDate(): self{
        $settings                                    = $this->integration_settings ?? [];
        $settings[self::IS_DELIVER_BY_DATE_REQUIRED] = true;
        $this->integration_settings                  = $settings;
        $this->save();
        return $this;
    }

    /**
     * @return $this
     */
    public function disableDeliverByDateCheck(): self{
        $settings                                    = $this->integration_settings ?? [];
        $settings[self::IS_DELIVER_BY_DATE_REQUIRED] = false;
        $this->integration_settings                  = $settings;
        $this->save();
        return $this;
    }

    /**
     * @return bool
     */
    public function shouldOrdersHaveDeliveryByDate(): bool{
        return isset($this->integration_settings[self::IS_DELIVER_BY_DATE_REQUIRED]) &&
            $this->integration_settings[self::IS_DELIVER_BY_DATE_REQUIRED] == true;
    }

    /**
     * @return string
     */
    public function getUsername(): string{
        return data_get($this->connection_settings, 'username');
    }

    /**
     * @return string
     */
    public function getPassword(): string{
        return data_get($this->connection_settings, 'password');
    }

    public function getSystemId(): string{
        return data_get($this->connection_settings, 'systemId');
    }


    /**
     * @param  array  $options
     * @return bool
     */
    public function save(array $options = [])
    {
        if(!$this->exists){
            // We enable only submitting orders with deliver by date
            $settings = $this->integration_settings;
            $settings[self::IS_DELIVER_BY_DATE_REQUIRED] = true;
            $settings[self::AUTOMATE_FULFILLMENT] = true;
            if(!isset($settings[self::GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD])){
                $settings[self::GIFT_CARD_NOTE_SALES_ORDER_CUSTOM_FIELD] = null;
            }
            $this->integration_settings = $settings;
        }
        return parent::save($options);
    }


    /**
     * @return void
     */
    protected static function boot()
    {
        static::addGlobalScope('veracore', function (Builder $query) {
            return $query->whereHas('integration', function (Builder $query) {
                $query->where('name', Integration::NAME_VERACORE);
            });
        });

        parent::boot();
    }
}
