<?php

namespace Modules\Shopify\Managers;

use App\Abstractions\Integrations\IntegrationInstanceInterface;
use App\Collections\ProductAttributeCollection;
use App\Collections\ProductImageCollection;
use App\Collections\ProductListingCollection;
use App\Collections\ProductPricingCollection;
use App\DTO\ProductAttributeDto;
use App\DTO\ProductDto;
use App\DTO\ProductImageDto;
use App\DTO\ProductListingDto;
use App\DTO\ProductListingWithProductIdDto;
use App\DTO\ProductPricingDto;
use App\DTO\SupplierDTO;
use App\Helpers;
use App\Importers\Parsers\FieldParserFactory;
use App\Integrations\Shopify;
use App\Models\Attribute;
use App\Models\DataImportMapping;
use App\Models\IntegrationInstance;
use App\Models\Product as SkuProduct;
use App\Models\ProductListing;
use App\Models\ProductPricingTier;
use App\Models\Setting;
use App\Models\Shopify\ShopifyProduct;
use Exception;
use Illuminate\Support\Collection;
use Str;

class ShopifyProductManager
{
    private mixed $mappings;

    private mixed $pricingMappings;

    private mixed $productAttributes;

    /**
     * @throws Exception
     */
    public function __construct(protected IntegrationInstance|IntegrationInstanceInterface $shopifyIntegrationInstance)
    {
    }

    /**
     * Apply the default mapping on data.
     *
     * @throws Exception
     */
    private function applyDataMapping(mixed $data, string $mappedColumn): mixed
    {
        $skuMapping = $this->mappings->where('sku_field', $mappedColumn)->first();
        $listingField = @$skuMapping['listing_field'];

        /*
         * TODO: I believe these are default mappings but the code doesn't make it obvious.  I would get default mappings
         *  in a separate method so that the method name makes it clear what is being done
         */

        if (is_null($listingField)) {
            switch ($mappedColumn) {
                case 'item_name':
                    $listingField = 'product_title';
                    break;

                case 'sales_channel_listing_id':
                    $listingField = 'variant_id';
                    break;

                case 'sku':
                    $listingField = 'variant_id';
                    break;

                case 'price':
                    $listingField = 'price';
                    break;

                case 'default_supplier':
                    $listingField = 'vendor';
                    break;
            }
        }

        //Return value
        $returnValue = @$data[$listingField];

        //Apply mappings
        if ($returnValue) {
            if (@$skuMapping['parsers'] && is_array($skuMapping['parsers'])) {
                foreach ($skuMapping['parsers'] as $parser) {
                    $returnValue = FieldParserFactory::make(
                        $parser['rule'],
                        $parser['args'],
                        $data
                    )->parse($returnValue);
                }
            }
        }

        return $returnValue;
    }

    public function makeProductListingCollectionForImport(Collection $shopifyProducts, bool $collectionHasProductIdColumn = false): ProductListingCollection
    {
        //Get amazon mappings
        $this->mappings = collect(DataImportMapping::query()
            ->where('model', 'listings')
            ->where('integration_instance_id', $this->shopifyIntegrationInstance->id)
            ->value('mapping'));

        $this->pricingMappings = $this->mappings
            ->filter(fn ($m) => preg_match("/price\..*\.value/", $m['sku_field']))
            ->map(fn ($m) => ['name' => explode('.', $m['sku_field'])[1], 'listing_field' => $m['listing_field']])
            ->keyBy('name');

        $this->productAttributes = $this->mappings
            ->filter(fn ($m) => Str::startsWith($m['sku_field'], 'attributes.'))
            ->map(fn ($m) => [
                'name' => str_replace('attributes.', '', $m['sku_field']),
                'listing_field' => $m['listing_field'],
            ])
            ->keyBy('name');

        //Make a collection for import
        $productListingCollection = new ProductListingCollection();
        $productPricingTiers = ProductPricingTier::all();
        $attributes = Attribute::all();

        foreach ($shopifyProducts as $record) {
            //Product Price Collection
            $productPriceCollection = new ProductPricingCollection();
            $productPricingTiers->whereIn('name', $this->pricingMappings->keys())
                ->each(function ($pricingTier) use (&$productPriceCollection, $record) {
                    if ($price = @$record->json_object[$this->pricingMappings[$pricingTier->name]['listing_field']]) {
                        $productPriceCollection->push(ProductPricingDto::from([
                            'sku' => $this->applyDataMapping($record, 'sku'),
                            'price' => $price,
                            'product_pricing_tier_id' => $pricingTier->id,
                        ]));
                    }
                });

            //Product Attributes Collection
            $productAttributeCollection = new ProductAttributeCollection();
            $attributes->whereIn('name', $this->productAttributes->keys())
                ->each(function ($attribute) use (&$productAttributeCollection, $record) {
                    if ($productAttribute = @$record->json_object[$this->productAttributes[$attribute->name]['listing_field']]) {
                        $productAttributeCollection->push(ProductAttributeDto::from([
                            'sku' => $this->applyDataMapping($record, 'sku'),
                            'value' => $productAttribute,
                            'attribute_id' => $attribute->id,
                        ]));
                    }
                });

            //Image collection
            $productImageCollection = new ProductImageCollection();
            if (
                ! is_null($record->images)
                && is_array($record->images)
            ) {
                foreach ($record->images as $image) {
                    $productImageCollection->push(ProductImageDto::from([
                        'url' => $image['src'],
                        'name' => $image['alt'],
                        'sku' => $this->applyDataMapping($record, 'sku'),
                    ]));
                }
            }

            $productListingDto = $collectionHasProductIdColumn ? ProductListingWithProductIdDto::class : ProductListingDto::class;
            $productDto = null;

            if (! $collectionHasProductIdColumn) {
                $productDto = ProductDto::from([
                    'sku' => $this->applyDataMapping($record, 'sku'),
                    'name' => $this->applyDataMapping($record, 'item_name'),
                    'type' => SkuProduct::TYPE_STANDARD,
                    // 'barcode'         => $this->applyDataMapping($record, 'barcode'),
                    'weight' => $this->applyDataMapping($record, 'weight') ?? 0,
                    // 'product_brand' => ProductBrandDto::from([
                    //     'name' => 'Brand Jatin', //TODO: Link
                    // ]),
                    'supplier' => SupplierDTO::from([
                        'name' => $this->applyDataMapping($record, 'default_supplier'),
                        'timezone' => Helpers::setting(Setting::KEY_DEFAULT_TIMEZONE, null, true),
                    ]),
                    'unit_cost' => null,
                    'images' => $productImageCollection,
                    'product_prices' => $productPriceCollection,
                    'product_attributes' => $productAttributeCollection,
                ]);
            }

            $productListingCollection->push($productListingDto::from([
                'document_id' => $record->id,
                'document_type' => ShopifyProduct::class,
                'listing_sku' => $record->variant_id,
                'title' => $this->applyDataMapping($record, 'item_name'),
                'sales_channel_listing_id' => $this->applyDataMapping($record, 'sales_channel_listing_id'),
                'sales_channel_id' => $this->shopifyIntegrationInstance->salesChannel->id,
                'product_id' => $collectionHasProductIdColumn ? $record->product_id : null,
                'product' => $productDto,
            ]));
        }

        return $productListingCollection;
    }

    public function resetInventory(ProductListing $productListing, string $locationId): void
    {
        if (\App::environment() !== 'production') {
            return;
        }
        $shopify = new Shopify($this->shopifyIntegrationInstance);
        $shopify->setInventoryLevel($productListing->shopifyListing->json_object['inventory_item_id'], $locationId, $productListing->quantity);
        $productListing->sales_channel_qty = $productListing->quantity;
        $productListing->save();
    }
}
