<?php

namespace Modules\ShipMyOrders\Tests\Feature;

use App\Exceptions\SalesOrder\SalesOrderFulfillmentDispatchException;
use App\Models\FinancialLine;
use App\Models\Integration;
use App\Models\IntegrationInstance;
use App\Models\NominalCode;
use App\Models\Product;
use App\Models\SalesOrder;
use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderLine;
use App\Models\User;
use App\Models\Warehouse;
use App\Services\SalesOrder\Fulfillments\FulfillmentManager;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Foundation\Testing\WithFaker;
use Illuminate\Support\Facades\Storage;
use Laravel\Sanctum\Sanctum;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoice;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceLine;
use Modules\ShipMyOrders\Managers\ShipMyOrdersInvoiceManager;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Tests\TestCase;
use Throwable;

class ShipMyOrderInvoiceControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use WithFaker;

    protected function setUp(): void
    {
        parent::setUp();

        $integration = Integration::query()->where('name',Integration::NAME_SHIPMYORDERS)->first();
        $warehouse = Warehouse::factory()->create(['type' => Warehouse::TYPE_3PL]);

        $this->instance = IntegrationInstance::factory()->create([
            'name' => Integration::NAME_SHIPMYORDERS,
            'connection_settings' => [
                'username' => $this->faker->md5(),
                'password' => $this->faker->md5(),
                'clientId' => $this->faker->word()
            ],
            'integration_id' => $integration ? $integration->id : Integration::factory()->create([
                'name' => Integration::NAME_SHIPMYORDERS,
                'integration_type' => Integration::TYPE_SHIPPING_PROVIDER
            ]),
            'integration_settings' => [
                'linked_warehouse_id' => $warehouse->id,
            ]
        ]);

        Sanctum::actingAs(User::first());
    }

    public function test_user_can_list_invoices(): void
    {
        ShipMyOrdersInvoice::factory()->count(3)->has(ShipMyOrdersInvoiceLine::factory()->count(5))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 3);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 15);

        $response = $this->getJson(route('shipmyorders.invoices.index'))->assertOk();

        $this->assertCount(3, $response->json()['data']);
    }

    public function test_user_can_view_invoice_details(): void
    {
        $invoice = ShipMyOrdersInvoice::factory()->has(ShipMyOrdersInvoiceLine::factory()->count(5))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 1);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 5);

        $response = $this->getJson(route('shipmyorders.invoices.show', ['invoice' => $invoice->id]))->assertOk();

        $this->assertNotNull($response->json()['data']);
        $this->assertCount(5, $response->json()['data']['items']);
    }

    /**
     * @throws SalesOrderFulfillmentDispatchException
     * @throws Throwable
     * @throws BindingResolutionException
     */
    public function test_user_can_upload_and_process_invoice()
    {
        $fileContents = '"NV","SIBER10495","5/31/2024","PO030085","Hourly warehouse/admin labor","40","3.5","140",""
"NV","SIBER10495","5/29/2024","PO030085","Additional Pallet","1.85","1","1.85",""
"NV","SIBER10495","6/3/2024","PO030085","Hourly warehouse/admin labor","40",".5","20",""
"NV","SIBER10495","6/3/2024","20240522_01","Hourly warehouse/admin labor","40","2.5","100",""
"NV","SIBER10495","6/3/2024","20240522_01","Boxes/Packaging Materials - Qtty: 41","94.14","1","94.14","7.888932"
"NV","SIBER10495","6/3/2024","SO-001.1","Fulfillment Fee","5.00","1","5.00","0.10"';

        $filename = 'invoice.csv';
        Storage::disk('smo-invoices')->put($filename, $fileContents);

        $response = $this->postJson(route('shipmyorders.invoices.store'), ['stored_name' => $filename])->assertSuccessful();
        $responseData = $response->json()['data'];
        $invoice = ShipMyOrdersInvoice::find($responseData['id']);

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 1);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 6);

        // Set up sales order and fulfillment

        $product = Product::factory()->create(['sku' => 'ABC']);

        $salesOrder = SalesOrder::factory()->create(['sales_order_number' => 'SO-001']);
        $salesOrderLine = SalesOrderLine::factory()->create([
            'sales_order_id' => $salesOrder->id,
            'product_id' => $product->id,
            'warehouse_id' => $this->instance->integration_settings['linked_warehouse_id']
        ]);
        $product->setInitialInventory($this->instance->integration_settings['linked_warehouse_id'], 10, 5.00);

        app(FulfillmentManager::class)->fulfill($salesOrder, [
            'fulfillment_type' => SalesOrderFulfillment::TYPE_MANUAL,
            'warehouse_id' => $this->instance->integration_settings['linked_warehouse_id'],
            'fulfillment_lines' => [
                [
                    'sales_order_line_id' => $salesOrderLine->id,
                    'quantity' => 1,
                ]
            ],
            false,
            false
        ]);

        $this->postJson(route('shipmyorders.invoices.process', ['invoice' => $invoice->id]))->assertSuccessful();

        $this->assertCount(1, ShipMyOrdersInvoice::whereNotNull('processed_at')->get());

        $this->assertDatabaseHas(FinancialLine::class, [
            'amount' => 5.00,
            'tax_allocation' => 0.10,
            'sales_order_id' => $salesOrder->id,
            'description' => 'Fulfillment Fee',
        ]);

        $this->assertDatabaseHas(ShipMyOrdersInvoiceLine::class, [
            'sales_order_fulfillment_id' => SalesOrderFulfillment::first()->id,
            'financial_line_id' => FinancialLine::first()->id,
        ]);

        $this->postJson(route('shipmyorders.invoices.unprocess', ['invoice' => $invoice->id]))->assertSuccessful();

        $this->assertDatabaseEmpty(FinancialLine::class);

        $this->assertCount(0, ShipMyOrdersInvoiceLine::whereNotNull('financial_line_id')->get());
    }

    public function test_user_can_archive_invoice(): void
    {
        ShipMyOrdersInvoice::factory()->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 1);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 2);

        $smoInvoice = ShipMyOrdersInvoice::first();

        $this->putJson(route('shipmyorders.invoices.archive', ['id' => $smoInvoice->id]))->assertSuccessful();

        $smoInvoice->refresh();

        $this->assertNotNull($smoInvoice->archived_at);
    }

    public function test_user_can_unarchive_invoice(): void
    {
        ShipMyOrdersInvoice::factory()->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create(['archived_at' => now()]);

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 1);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 2);

        $smoInvoice = ShipMyOrdersInvoice::first();

        $this->assertNotNull($smoInvoice->archived_at);

        $this->putJson(route('shipmyorders.invoices.unarchived', ['id' => $smoInvoice->id]))->assertSuccessful();

        $smoInvoice->refresh();

        $this->assertNull($smoInvoice->archived_at);
    }

    public function test_user_can_bulk_archive_invoices(): void
    {
        ShipMyOrdersInvoice::factory()->count(3)->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 3);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 6);

        $smoInvoices = ShipMyOrdersInvoice::all();

        $this->putJson(route('shipmyorders.invoices.bulk-archive'), [
            'ids' => $smoInvoices->pluck('id')->toArray()
        ])->assertSuccessful();

        ShipMyOrdersInvoice::all()->each(function (ShipMyOrdersInvoice $invoice) {
            $this->assertNotNull($invoice->archived_at);
        });
    }

    public function test_user_can_bulk_unarchive_invoices(): void
    {
        ShipMyOrdersInvoice::factory()->count(3)->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 3);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 6);

        $smoInvoices = ShipMyOrdersInvoice::all();

        $this->putJson(route('shipmyorders.invoices.bulk-unarchive'), [
            'ids' => $smoInvoices->pluck('id')->toArray()
        ])->assertSuccessful();

        ShipMyOrdersInvoice::all()->each(function (ShipMyOrdersInvoice $invoice) {
            $this->assertNull($invoice->archived_at);
        });
    }

    public function test_user_can_delete_invoice(): void
    {
        ShipMyOrdersInvoice::factory()->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 1);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 2);

        $smoInvoice = ShipMyOrdersInvoice::first();

        $this->deleteJson(route('shipmyorders.invoices.destroy', ['invoice' => $smoInvoice->id]))->assertSuccessful();

        $this->assertDatabaseEmpty(ShipMyOrdersInvoice::class);
        $this->assertDatabaseEmpty(ShipMyOrdersInvoiceLine::class);
    }

    public function test_user_can_bulk_delete_invoices(): void
    {
        ShipMyOrdersInvoice::factory()->count(3)->has(ShipMyOrdersInvoiceLine::factory()->count(2))->create();

        $this->assertDatabaseCount(ShipMyOrdersInvoice::class, 3);
        $this->assertDatabaseCount(ShipMyOrdersInvoiceLine::class, 6);

        $smoInvoices = ShipMyOrdersInvoice::all();

        $this->deleteJson(route('shipmyorders.invoices.bulk-destroy'), [
            'ids' => $smoInvoices->pluck('id')->toArray()
        ])->assertSuccessful();

        $this->assertDatabaseEmpty(ShipMyOrdersInvoice::class);
        $this->assertDatabaseEmpty(ShipMyOrdersInvoiceLine::class);
    }

    public function test_user_can_update_invoice(): void
    {
        $invoice = ShipMyOrdersInvoice::factory()->create();

        $line = ShipMyOrdersInvoiceLine::factory()->create([
            'ship_my_orders_invoice_id' => $invoice->id,
            'nominal_code_id' => null
        ]);

        $this->assertNull($line->nominal_code_id);

        $invoice = $invoice->toArray();

        $edited = $line->toArray();
        $edited['nominal_code_id'] = NominalCode::factory()->create()->id;

        $invoice['lines'] = [$edited];

        $this->putJson(route('shipmyorders.invoices.update', ['invoice' => $invoice['id']]), $invoice)->assertSuccessful();

        $line->refresh();

        $this->assertNotNull($line->nominal_code_id);
    }
}