<?php

namespace Modules\ShipMyOrders\Repositories;

use App\Models\SalesOrderFulfillment;
use App\Models\ShippingMethod;
use App\Services\ShippingProvider\ShippingProviderOrder;
use App\Services\ShippingProvider\ShippingProviderRepository;
use Illuminate\Database\Eloquent\Model;
use Modules\ShipMyOrders\Entities\ShipMyOrdersOrder;

class ShipMyOrdersOrderRepository implements ShippingProviderRepository
{
    /**
     * @param  int  $fulfillmentId
     * @return ShippingProviderOrder|Model|null
     */
    public function getOrderByFulfillmentId(int $fulfillmentId): ShippingProviderOrder|Model|null
    {
        return ShipMyOrdersOrder::query()->where('sku_fulfillment_id', $fulfillmentId)->first();
    }

    /**
     * @param  int|string  $response
     * @param  SalesOrderFulfillment  $fulfillment
     * @return ShippingProviderOrder|Model
     */
    public function updateOrCreateByResponse(int|string $response, SalesOrderFulfillment $fulfillment): ShippingProviderOrder|Model
    {
        return ShipMyOrdersOrder::query()
            ->firstOrCreate(['smo_id' => $response], [
                'json_data' => json_decode($response, 1),
                'reference_number' => $fulfillment->fulfillment_number,
                'sku_fulfillment_id' => $fulfillment->id,
            ]);
    }

    /**
     * @return array
     */
    public function getOrderIdsNeedingTrackingInfo(): array
    {
        return ShipMyOrdersOrder::query()
            ->whereNull('json_data')
            ->orWhereRaw("JSON_EXTRACT(json_data, '$.Shipping.TrackingNumber') IS NULL")
            ->pluck('reference_number')
            ->all();
    }

    /**
     * @param  array  $orders
     * @return array
     */
    public function updateAll(array $orders): array
    {
        $ids = [];
        foreach ($orders as $order) {
            $smoOrder = ShipMyOrdersOrder::query()->where('reference_number', $order['OrderID'])->first();
            $jsonData = $smoOrder->json_data;
            $jsonData = array_merge($jsonData, $order);
            $smoOrder->json_data = $jsonData;
            $smoOrder->save();
            $ids[] = $smoOrder->smo_id;
        }

        return $this->findByIds($ids);
    }

    /**
     * @param  array  $ids
     * @return array
     */
    public function findByIds(array $ids): array
    {
        return ShipMyOrdersOrder::query()->whereIn('smo_id', $ids)->get()->all();
    }

    /**
     * @param  array  $orders
     * @return array
     */
    public function getShippingMethodsIn(array $orders): array
    {
        $shippingMethodNames = array_unique(
            array_map(
                fn($order) => $order['ShippingMethod'] ?? '',
                $orders
            )
        );

        return ShippingMethod::query()
            ->whereIn('full_name', $shippingMethodNames)->get()->toArray();
    }

    /**
     * @param  array  $fulfillmentData
     * @return int
     */
    public function updateFulfillmentsWithTrackingInfo(array $fulfillmentData): int
    {
        return batch()->update(new SalesOrderFulfillment, $fulfillmentData, 'id');
    }

    /**
     * @param  array  $orders
     * @return array
     */
    public function getFulfillmentsForOrders(array $orders): array
    {
        return SalesOrderFulfillment::query()
            ->select('sales_order_fulfillments.*')
            ->addSelect('smo_id as shipping_provider_id')
            ->join('ship_my_orders_orders', 'ship_my_orders_orders.sku_fulfillment_id', 'sales_order_fulfillments.id')
            ->whereIn('ship_my_orders_orders.smo_id', collect($orders)->pluck('smo_id')->toArray())
            ->get()->all();
    }


}