<?php

namespace Modules\ShipMyOrders\Managers;

use App\Models\SalesOrderFulfillment;
use App\Models\SalesOrderFulfillmentLine;
use App\Services\ShippingProvider\ShippingProviderOrderManager;
use App\Utility\PhoneNumberUtility;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Str;
use Modules\ShipMyOrders\Entities\ShipMyOrdersIntegrationInstance;
use Modules\ShipMyOrders\Repositories\ShipMyOrdersInventoryRepository;
use Modules\ShipMyOrders\Repositories\ShipMyOrdersOrderRepository;
use Modules\ShipMyOrders\Services\ShipMyOrdersClient;

class ShipMyOrdersOrderManager extends ShippingProviderOrderManager
{

    /**
     * @param  ShipMyOrdersOrderRepository  $orders
     */
    public function __construct(
        private readonly ShipMyOrdersOrderRepository $orders,
        private readonly ShipMyOrdersInventoryRepository $inventory,
    ){
        parent::__construct(
            instance: $instance = ShipMyOrdersIntegrationInstance::active(),
            client: new ShipMyOrdersClient($instance),
            orders: $this->orders
        );
    }

    /**
     * @param  SalesOrderFulfillment  $fulfillment
     * @return array
     */
    public function makeFulfillmentOrderPayload(SalesOrderFulfillment $fulfillment): array
    {
        $salesOrder = $fulfillment->salesOrder;
        $shipToAddress = $salesOrder->shippingAddress;

        if($fulfillment->requestedShippingMethod?->full_name === null){
            $shippingMethod = '';
        } else {
            // Translate the shipping method
            $shippingMethod = ShipMyOrdersShippingMethodMapperManager::translateToShipMyOrdersShippingMethod(
                $fulfillment->requestedShippingMethod?->full_name
            );
        }

        // Prep the order lines
        $orderLines = $fulfillment->salesOrderFulfillmentLines;

        /** @var Collection $preparedOrderLines */
        $preparedOrderLines = $orderLines->map(function (SalesOrderFulfillmentLine $line) {
            return [
                'productSku' => $line->salesOrderLine->product?->sku,
                'productName' => $line->salesOrderLine->product?->name,
                'orderQuantity' => $line->quantity,
                'declaredValue' => $line->salesOrderLine->salesOrderLineFinancial?->cogs ?? 0,
            ];
        });

        /** @var float $totalOrderValue */
        $totalOrderValue = $preparedOrderLines->reduce(function ($carry, $item) {
            return $carry + ($item['declaredValue'] * $item['orderQuantity']);
        }, 0.0);

        if (strtolower(app()->environment()) === 'production') {
            $orderPrefix = '';
        } else {
            $orderPrefix = 'TEST_';
        }

        return [
            'orderId' => $orderPrefix.$fulfillment->fulfillment_number,
            'orderDate' => $salesOrder->order_date->format("m/d/Y"),
            'orderLines' => $preparedOrderLines,
            'shipToCompany' => $shipToAddress->company,
            'shipToFullName' => $shipToAddress->name,
            'shipToAddressLine1' => $shipToAddress->address1,
            'shipToAddressLine2' => $shipToAddress->address2,
            'shipToCity' => $shipToAddress->city,
            'shipToState' => $shipToAddress->province ?? $shipToAddress->province_code,
            'shipToZip' => Str::substr($shipToAddress->zip, 0, 5),
            'shipToCountry' => $shipToAddress->country_code ?? $shipToAddress->country,
            'shipToPhone' => PhoneNumberUtility::applyMany(
                $shipToAddress->phone,
                ['digitsOnly', 'withoutLeadingOne']
            ),
            'shipToEmailAddress' => null,
            'shippingMethod' => $shippingMethod,
            'isSignatureRequired' => $fulfillment->metadata['isSignatureRequired'] ?? 0,
            'isSaturdayDelivery' => 0,
            'isInsured' => $fulfillment->metadata['isInsured'] ?? 0,
            'orderInsuranceValue' => $totalOrderValue, // N/A
            'hasAttachedDocs' => 0,
            'packingSlipMessage' => null,
            'instructionsForWarehouse' => $fulfillment->metadata['instructionsForWarehouse'] ?? null,
            'usingThirdPartyAccount' => null,
            'thirdPartyAccountNumber' => null,
            'thirdPartyBillingZip' => null,
        ];
    }

    public function refreshInventory(): void
    {
        $data = $this->client->getInventory();

        $this->inventory->save($data);
    }

    public function reportInventoryDiscrepancies(): array
    {
        return $this->inventory->inventoryDiscrepancies()->toArray();
    }
}