<?php

namespace Modules\ShipMyOrders\Managers;

use App\Models\NominalCode;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceLine;
use Modules\ShipMyOrders\Entities\ShipMyOrdersInvoiceNominalCodeMappingRule;
use Modules\ShipMyOrders\Exceptions\NotSupportedOperatorException;

class ShipMyOrdersInvoiceNominalCodeMappingRuleManager
{
    /**
     * @throws NotSupportedOperatorException
     */
    public function getNominalCodeForInvoiceLine(ShipMyOrdersInvoiceLine $line): ?NominalCode
    {
        $line->loadMissing('salesOrderFulfillment.salesOrder');

        $rules = ShipMyOrdersInvoiceNominalCodeMappingRule::with('nominalCode')->orderBy('sort_order')->get();

        foreach ($rules as $rule) {
            if ($this->checkConditions($rule, $line)) {
                return $rule->nominalCode;
            }
        }

        return null;
    }

    /**
     * @throws NotSupportedOperatorException
     */
    private function checkConditions(ShipMyOrdersInvoiceNominalCodeMappingRule $rule, ShipMyOrdersInvoiceLine $line): bool
    {
        foreach ($rule->conditions as $condition) {
            switch (strtolower($condition['column'])) {
                case 'description':
                case 'quantity':
                    $status = $this->checkOperatorCondition($condition, $line);
                    break;
                case 'salesorder':
                    $condition['column'] = 'salesOrderFulfillment.salesOrder';
                    $status              = $this->checkRelationCondition($condition, $line);
                    break;
                default:
                    return false;
            }
            if (!$status) {
                return false;
            }
        }

        return true;
    }

    /**
     * @throws NotSupportedOperatorException
     */
    private function checkOperatorCondition(array $condition, ShipMyOrdersInvoiceLine $line): bool
    {
        $condition['column'] = strtolower($condition['column']);
        return match ($condition['operator']) {
            '=' => $line[$condition['column']] == $condition['value'],
            '!=' => $line[$condition['column']] != $condition['value'],
            '>' => $line[$condition['column']] > $condition['value'],
            '>=' => $line[$condition['column']] >= $condition['value'],
            '<' => $line[$condition['column']] < $condition['value'],
            '<=' => $line[$condition['column']] <= $condition['value'],
            'contains' => str_contains(strtolower($line[$condition['column']]), strtolower($condition['value'])),
            'not contains' => !str_contains(strtolower($line[$condition['column']]), strtolower($condition['value'])),
            'in' => in_array($line[$condition['column']], explode(',', $condition['value'])),
            'not in' => !in_array($line[$condition['column']], explode(',', $condition['value'])),
            default => throw new NotSupportedOperatorException(),
        };
    }

    /**
     * @throws NotSupportedOperatorException
     */
    private function checkRelationCondition(array $condition, ShipMyOrdersInvoiceLine $line): bool
    {
        return match ($condition['operator']) {
            'exists' => isset($line['salesOrderFulfillment']['salesOrder']),
            'not exists' => !isset($line['salesOrderFulfillment']['salesOrder']),
            default => throw new NotSupportedOperatorException(),
        };
    }
}