<?php

namespace Modules\ShipMyOrders\Entities;

use App\Abstractions\UniqueFieldsInterface;
use App\Models\Contracts\Filterable;
use App\Models\FinancialLine;
use App\Models\NominalCode;
use App\Models\SalesOrderFulfillment;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Modules\ShipMyOrders\Database\factories\ShipMyOrdersInvoiceLineFactory;

/**
 * @property int $id
 * @property int $ship_my_orders_invoice_id
 * @property int $sales_order_fulfillment_id
 * @property int $financial_line_id
 * @property Carbon $date
 * @property string $reference
 * @property string $description
 * @property float $amount
 * @property int $quantity
 * @property float $subtotal
 * @property float $tax
 * @property float $total
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 * @property ?Carbon $processed_at
 * @property-read ShipMyOrdersInvoice $shipMyOrdersInvoice
 * @property-read SalesOrderFulfillment $salesOrderFulfillment
 * @property-read FinancialLine $financialLine
 *
 */
class ShipMyOrdersInvoiceLine extends Model implements UniqueFieldsInterface, Filterable
{
    use HasFactory;

    protected $casts = [
        'date' => 'date',
        'processed_at' => 'datetime'
    ];

    protected $fillable = [
        'ship_my_orders_invoice_id',
        'sales_order_fulfillment_id',
        'financial_line_id',
        'date',
        'reference',
        'description',
        'amount',
        'quantity',
        'subtotal',
        'tax',
        'total',
        'processed_at'
    ];

    public static function getUniqueFields(): array
    {
        return ['ship_my_orders_invoice_id', 'reference', 'description', 'amount', 'quantity'];
    }

    public static function newFactory(): Factory
    {
        return ShipMyOrdersInvoiceLineFactory::new();
    }

    /*
    |--------------------------------------------------------------------------
    | Relations
    |--------------------------------------------------------------------------
    */

    public function shipMyOrdersInvoice(): BelongsTo
    {
        return $this->belongsTo(ShipMyOrdersInvoice::class);
    }

    public function salesOrderFulfillment(): BelongsTo
    {
        return $this->belongsTo(SalesOrderFulfillment::class);
    }

    public function financialLine(): BelongsTo
    {
        return $this->belongsTo(FinancialLine::class);
    }

    public function nominalCode(): BelongsTo
    {
        return $this->belongsTo(NominalCode::class);
    }

    public function availableColumns()
    {
        return [
            'id' => 'integer',
            'ship_my_orders_invoice_id' => 'integer',
            'date' => 'date',
            'reference' => 'string',
            'description' => 'string',
            'amount' => 'decimal',
            'quantity' => 'decimal',
            'subtotal' => 'decimal',
            'tax' => 'decimal',
            'total' => 'decimal',
            'processed_at' => 'date'
        ];
    }

    public function filterableColumns(): array
    {
        return $this->availableColumns();
    }

    public function generalFilterableColumns(): array
    {
        return ['reference'];
    }
}
