<?php

namespace Modules\ShipMyOrders\Entities;

use App\Abstractions\UniqueFieldsInterface;
use App\DataTable\DataTableModelInterface;
use App\DataTable\DataTableModelTrait;
use App\Models\Concerns\Archive;
use App\Models\Concerns\HasFilters;
use App\Models\Concerns\HasSort;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Factories\Factory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Modules\ShipMyOrders\Database\factories\ShipMyOrdersInvoiceFactory;
use Modules\ShipMyOrders\Managers\ShipMyOrdersInvoiceManager;
use Throwable;

/**
 * @property int $id
 * @property string $invoice_number
 * @property string $invoice_date
 * @property float $total
 * @property Carbon $created_at
 * @property ?Carbon $updated_at
 * @property ?Carbon $archived_at
 * @property ?Carbon $processed_at
 * @property-read ShipMyOrdersInvoiceLine[] $shipMyOrdersInvoiceLines
 *
 */
class ShipMyOrdersInvoice extends Model implements UniqueFieldsInterface, DataTableModelInterface
{
    use HasFilters;
    use HasSort;
    use HasFactory;
    use Archive;
    use DataTableModelTrait;

    protected $casts = [
        'invoice_date' => 'date',
        'total' => 'float',
        'archived_at' => 'datetime',
        'processed_at' => 'datetime',
    ];

    protected $fillable = [
        'invoice_number',
        'invoice_date',
        'total',
        'processed_at'
    ];

    public static function getUniqueFields(): array
    {
        return ['invoice_number'];
    }


    public function shipMyOrdersInvoiceLines(): HasMany
    {
        return $this->hasMany(ShipMyOrdersInvoiceLine::class);
    }

    /**
     * @throws Throwable
     */
    public function delete(): ?bool
    {
        if ($this->isUsed()) {
            return true;
        }

        app(ShipMyOrdersInvoiceManager::class)->unprocessInvoice($this);

        $this->shipMyOrdersInvoiceLines()->delete();
        return parent::delete();
    }

    public function isUsed(): bool
    {
        return false;
    }

    public function visibleColumns(): array
    {
        return array_filter(array_keys($this->availableColumns()), fn($value) => !in_array($value, ['created_at', 'updated_at']));
    }

    public static function specialLabels(): array
    {
        return [
            'invoice_number' => 'Invoice #',
            'id' => 'ID'
        ];
    }

    public function generalFilterableColumns(): array
    {
        return ['invoice_number'];
    }

    public static function newFactory(): Factory
    {
        return ShipMyOrdersInvoiceFactory::new();
    }
}
