<?php

namespace Modules\Qbo\Tests\Feature;

use App\Models\AccountingTransaction;
use App\Models\AccountingTransactionLine;
use App\Models\NominalCode;
use App\Models\Payment;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseInvoiceLine;
use App\Models\PurchaseOrder;
use App\Models\PurchaseOrderLine;
use App\Models\SalesChannel;
use App\Models\SalesCredit;
use App\Models\SalesCreditLine;
use App\Models\SalesOrder;
use App\Models\SalesOrderLine;
use App\Models\StockTake;
use App\Models\StockTakeItem;
use App\Models\Supplier;
use App\Models\TaxRate;
use App\Models\User;
use Http;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Modules\Qbo\ApiParameterObjects\QboQueryApo;
use Modules\Qbo\Entities\QboAccount;
use Modules\Qbo\Entities\QboBill;
use Modules\Qbo\Entities\QboCreditMemo;
use Modules\Qbo\Entities\QboIntegrationInstance;
use Modules\Qbo\Entities\QboInvoice;
use Modules\Qbo\Entities\QboItem;
use Modules\Qbo\Entities\QboJournal;
use Modules\Qbo\Entities\QboPayment;
use Modules\Qbo\Entities\QboPurchaseOrder;
use Modules\Qbo\Entities\QboTaxCode;
use Modules\Qbo\Enums\QboAccountTypeEnum;
use Modules\Qbo\Jobs\QboDeletePaymentsJob;
use Modules\Qbo\Jobs\QboUpdateOrCreateBillsJob;
use Modules\Qbo\Jobs\QboUpdateOrCreateCreditMemosJob;
use Modules\Qbo\Jobs\QboUpdateOrCreateInvoicesJob;
use Modules\Qbo\Jobs\QboUpdateOrCreateJournalsJob;
use Modules\Qbo\Jobs\QboUpdateOrCreatePaymentsJob;
use Modules\Qbo\Jobs\QboUpdateOrCreatePurchaseOrdersJob;
use Modules\Qbo\Manager\QboManager;
use Modules\Qbo\Tests\QboMockRequests;
use Plannr\Laravel\FastRefreshDatabase\Traits\FastRefreshDatabase;
use Str;
use Tests\TestCase;

class QboTransactionControllerTest extends TestCase
{
    use FastRefreshDatabase;
    use QboMockRequests;

    private bool $mockEnabled = true;

    private QboIntegrationInstance $qboIntegrationInstance;

    private TaxRate $defaultTaxRate;

    public function setUp(): void
    {
        parent::setUp();

        //Create QBO instance
        $this->qboIntegrationInstance = QboIntegrationInstance::factory()
            ->has(SalesChannel::factory())
            ->create();

        if ($this->mockEnabled) {
            $this->mockGetAccessToken();
        }

        Queue::fake();
        Sanctum::actingAs(User::first());

        if ($this->mockEnabled) {
            Http::preventStrayRequests();
        }

        if (! $this->mockEnabled) {
            (new QboManager($this->qboIntegrationInstance))->refreshAccounts(new QboQueryApo());
            (new QboManager($this->qboIntegrationInstance))->createNominalCodes(QboAccount::all()->pluck('id')->values()->toArray());

            //This is used in linking product's Income and Expense account
            NominalCode::where('name', QboAccountTypeEnum::COST_OF_GOODS_SOLD->value)->update([
                'code' => QboAccount::where('Classification', 'Expense')->first()->QboId,
            ]);
        } else {
            $qboAccount = QboAccount::factory()->create([
                'integration_instance_id' => $this->qboIntegrationInstance->id,
                'json_object' => [
                    'Id' => 1,
                    'Name' => 'Qbo Testing', //Added for a reason to make the test pass with actual calls.
                    'AccountType' => 'Expense',
                ],
            ]);

            (new QboManager($this->qboIntegrationInstance))->createNominalCodes([$qboAccount->id]);
        }

        // Link tax rate
        $this->defaultTaxRate = TaxRate::latest()->first();
        if (! $this->mockEnabled) {
            (new QboManager($this->qboIntegrationInstance))->refreshTaxRates(new QboQueryApo());
            (new QboManager($this->qboIntegrationInstance))->refreshTaxCodes(new QboQueryApo());

            $this->defaultTaxRate->accounting_integration_id = QboTaxCode::orderBy('id', 'desc')->first()->id;
            $this->defaultTaxRate->accounting_integration_type = QboTaxCode::class;
            $this->defaultTaxRate->update();
        }

        // Create default tax rate
        if ($this->mockEnabled) {
            //TODO: Replace with factory
            QboItem::create([
                'integration_instance_id' => $this->qboIntegrationInstance->id,
                'json_object' => [
                    'Name' => QboItem::TAX_PAYABLE_ITEM,
                    'Id' => 100,
                    'IncomeAccountRef' => [
                        'value' => 1,
                        'name' => 'test',
                    ],
                    'ExpenseAccountRef' => [
                        'value' => 1,
                        'name' => 'test',
                    ],
                ],
            ]);
        }
    }

    public function test_qbo_it_can_sync_qbo_invoices_and_payments()
    {
        /*
        |--------------------------------------------------------------------------
        | Sync purchase orders
        |--------------------------------------------------------------------------
        */

        // Create Purchase Order
        $purchaseOrder = PurchaseOrder::factory()->hasPurchaseOrderLines()->for(Supplier::query()->first())->create([
            'tax_rate_id' => $this->defaultTaxRate->id,
            'supplier_id' => Supplier::query()->first()->id,
        ]);
        $accountingTransaction = AccountingTransaction::factory()->create([
            'type' => AccountingTransaction::TYPE_PURCHASE_ORDER,
            'link_id' => $purchaseOrder->id,
            'link_type' => PurchaseOrder::class,
            'reference' => $purchaseOrder->purchase_order_number,
        ]);

        // Create Accounting Transactions
        AccountingTransactionLine::factory()
            ->for($accountingTransaction)
            ->create([
                'link_id' => $purchaseOrder->purchaseOrderLines->first()->id,
                'link_type' => PurchaseOrderLine::class,
                'type' => AccountingTransactionLine::TYPE_PURCHASE_ORDER_LINE,
                'nominal_code_id' => NominalCode::where('type', 'Expense')->first()->id,
                'quantity' => $purchaseOrder->purchaseOrderLines->first()->quantity,
            ]);

        if ($this->mockEnabled) {
            $this->mockUpdateOrCreatePurchaseOrders(PurchaseOrder::all());
            $this->mockCreateVendor($purchaseOrder->supplier);
            $this->mockCreateItem($purchaseOrder->PurchaseOrderLines->first()->product);
        }

        // Assert Create
        $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreatePurchaseOrdersJob::class)->first()->handle();
        $this->assertEquals(1, QboPurchaseOrder::query()->count());
        $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        // Assert Update
        if (! $this->mockEnabled) {
            $qboPurchaseOrder = QboPurchaseOrder::query()->first();
            AccountingTransaction::where([
                'accounting_integration_id' => $qboPurchaseOrder->id,
                'accounting_integration_type' => QboPurchaseOrder::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);

            $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreatePurchaseOrdersJob::class)[1]->handle();
            $this->assertEquals(1, QboPurchaseOrder::where('updated_at', '>', $qboPurchaseOrder->updated_at)->count());
            $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Sync Purchase Order Invoice / Bills
        |--------------------------------------------------------------------------
        */

        //Create Purchase Order Invoice
        $purchaseInvoice = PurchaseInvoice::factory()->hasPurchaseInvoiceLines()->create([
            'supplier_id' => Supplier::query()->first()->id,
            'purchase_order_id' => $purchaseOrder->id,
        ]);

        $accountingTransaction = AccountingTransaction::factory()->create([
            'type' => AccountingTransaction::TYPE_PURCHASE_ORDER_INVOICE,
            'link_id' => $purchaseInvoice->id,
            'link_type' => PurchaseInvoice::class,
            'reference' => $purchaseInvoice->supplier_invoice_number,
        ]);

        //Create Accounting Transactions
        AccountingTransactionLine::factory()
            ->for($accountingTransaction)
            ->create([
                'link_id' => $purchaseInvoice->purchaseInvoiceLines->first()->id,
                'link_type' => PurchaseInvoiceLine::class,
                'type' => AccountingTransactionLine::TYPE_PURCHASE_ORDER_LINE,
                'nominal_code_id' => NominalCode::where('type', 'Expense')->first()->id,
                'quantity' => $purchaseInvoice->purchaseInvoiceLines->first()->quantity_invoiced,
            ]);

        if ($this->mockEnabled) {
            $this->clearExistingFakes();
            $this->mockUpdateOrCreateBills(PurchaseInvoice::all());
            $this->mockCreateVendor($purchaseInvoice->supplier);
        }

        // Create Bill
        $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreateBillsJob::class)->first()->handle();
        $this->assertEquals(1, QboBill::query()->count());
        $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        // Update Bill
        if (! $this->mockEnabled) {
            $qboBill = QboBill::query()->first();
            AccountingTransaction::where([
                'accounting_integration_id' => $qboBill->id,
                'accounting_integration_type' => QboBill::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);
            $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreateBillsJob::class)[1]->handle();
            $this->assertEquals(1, QboBill::where('updated_at', '>', $qboBill->updated_at)->count());
            $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Refresh Sales Orders / Invoices
        |--------------------------------------------------------------------------
        */

        //Create Sales Order
        $salesOrder = SalesOrder::factory()->create([
            'total_revenue' => 100,
            'tax_rate_id' => $this->defaultTaxRate->id,
        ]);
        $salesOrderLine = SalesOrderLine::factory()->create([
            'sales_order_id' => $salesOrder->id,
            'amount' => 100,
            'quantity' => 1,
        ]);
        $salesOrderLine = SalesOrderLine::with('product')->where('sales_order_id', $salesOrder->id)->first();

        $accountingTransaction = AccountingTransaction::factory()->create([
            'type' => AccountingTransaction::TYPE_SALES_ORDER_INVOICE,
            'link_id' => $salesOrder->id,
            'link_type' => SalesOrder::class,
            'reference' => $salesOrder->sales_order_number,
            'total' => $salesOrder->total_revenue,
        ]);

        //Create Accounting Transactions
        AccountingTransactionLine::factory()
            ->for($accountingTransaction)
            ->create([
                'amount' => $salesOrder->total_revenue,
                'link_id' => $salesOrder->salesOrderLines->first()->id,
                'link_type' => SalesOrderLine::class,
                'type' => AccountingTransactionLine::TYPE_SALES_INVOICE_LINE,
                'quantity' => $salesOrder->salesOrderLines->first()->quantity,
            ]);

        if ($this->mockEnabled) {
            //TODO: Remove this after fixing QboCreateItemApo
            QboAccount::factory()->create([
                'integration_instance_id' => $this->qboIntegrationInstance->id,
                'json_object' => [
                    'Classification' => 'Revenue',
                    'Id' => 100,
                ],
            ]);
            $this->clearExistingFakes();
            $this->mockUpdateOrCreateInvoices(SalesOrder::all());
            $this->mockCreateCustomer($salesOrder->customer);
            $this->mockCreateItem($salesOrderLine->product);
        }

        // Create Invoice
        $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreateInvoicesJob::class)->first()->handle();
        $this->assertEquals(1, QboInvoice::query()->count());
        $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        // Update Invoice
        if (! $this->mockEnabled) {
            $qboInvoice = QboInvoice::query()->first();
            AccountingTransaction::where([
                'accounting_integration_id' => $qboInvoice->id,
                'accounting_integration_type' => QboInvoice::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);
            $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreateInvoicesJob::class)[1]->handle();
            $this->assertEquals(1, QboInvoice::where('updated_at', '>', $qboInvoice->updated_at)->count());
            $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Sync Sales Order Payments
        |--------------------------------------------------------------------------
        */
        //Create payment for order
        Payment::factory()->create([
            'link_type' => SalesOrder::class,
            'link_id' => $salesOrder->id,
            'amount' => $salesOrder->total_revenue,
            'currency_rate' => 1,
            'external_reference' => Str::random(21), // Added because quickbooks do not allow more than this
        ]);

        if ($this->mockEnabled) {
            $this->clearExistingFakes();
            $this->mockUpdateOrCreatePayments(QboInvoice::all(), Payment::all());
        }

        // Create payment
        $this->postJson(route('qbo.transactions.syncPayments', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreatePaymentsJob::class)->first()->handle();
        $this->assertEquals(1, QboPayment::query()->count());
        $this->assertEquals(0, Payment::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        //Update payment
        if (! $this->mockEnabled) {
            $qboPayment = QboPayment::query()->first();
            Payment::where([
                'accounting_integration_id' => $qboPayment->id,
                'accounting_integration_type' => QboPayment::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);
            $this->postJson(route('qbo.transactions.syncPayments', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreatePaymentsJob::class)->last()->handle();
            $this->assertEquals(1, QboPayment::where('updated_at', '>', $qboPayment->updated_at)->count());
            $this->assertEquals(0, Payment::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Sync Journals
        |--------------------------------------------------------------------------
        */

        $stockTake = StockTake::factory()->create();
        StockTakeItem::factory()->create([
            'stock_take_id' => $stockTake->id,
        ]);
        $stockTakeAccountingTransaction = AccountingTransaction::factory()->create([
            'reference' => 'Stock Take '.$stockTake->id.' for SMO_1',
            'link_id' => $stockTake->id,
            'link_type' => StockTake::class,
            'type' => AccountingTransaction::TYPE_STOCK_TAKE,
        ]);

        StockTakeItem::all()
            ->each(function (StockTakeItem $stockTakeItem) use ($stockTakeAccountingTransaction) {
                AccountingTransactionLine::factory()->create([
                    'type' => AccountingTransactionLine::TYPE_CREDIT,
                    'nominal_code_id' => NominalCode::where('type', 'Expense')->first()->id,
                    'accounting_transaction_id' => $stockTakeAccountingTransaction->id,
                    'link_id' => $stockTakeItem->id,
                    'link_type' => StockTakeItem::class,
                    'amount' => 102,
                    'quantity' => 1,
                ]);

                AccountingTransactionLine::factory()->create([
                    'type' => AccountingTransactionLine::TYPE_DEBIT,
                    'nominal_code_id' => NominalCode::where('type', 'Expense')->first()->id,
                    'accounting_transaction_id' => $stockTakeAccountingTransaction->id,
                    'link_id' => $stockTakeItem->id,
                    'link_type' => StockTakeItem::class,
                    'amount' => 102,
                    'quantity' => 1,
                ]);
            });

        if ($this->mockEnabled) {
            $this->clearExistingFakes();
            $this->mockUpdateOrCreateJournals($stockTakeAccountingTransaction);
        }

        $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreateJournalsJob::class)->first()->handle();
        $this->assertEquals(1, QboJournal::query()->count());
        $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        if (! $this->mockEnabled) {
            $qboJournal = QboJournal::query()->first();
            AccountingTransaction::where([
                'accounting_integration_id' => $qboJournal->id,
                'accounting_integration_type' => QboJournal::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);
            $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreateJournalsJob::class)[1]->handle();
            $this->assertEquals(1, QboJournal::where('updated_at', '>', $qboJournal->updated_at)->count());
            $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Sync Credit Memos
        |--------------------------------------------------------------------------
        */

        $salesCredit = SalesCredit::factory()
            ->create([
                'sales_order_id' => $salesOrder->id,
                'customer_id' => $salesOrder->customer->id,
                'total_credit' => $salesOrder->total_revenue,
            ]);
        $salesOrderLine = $salesOrder->salesOrderLines->first();
        $salesCreditLine = SalesCreditLine::factory()->create([
            'sales_order_line_id' => $salesOrderLine->id,
            'sales_credit_id' => $salesCredit->id,
            'product_id' => $salesOrderLine->product->id,
            'quantity' => $salesOrderLine->quantity,
            'amount' => $salesOrderLine->accountingTransactionLine->amount,
            'unit_cost' => $salesOrderLine->unit_cost,
        ]);

        $accountingTransaction = AccountingTransaction::factory()->create([
            'total' => $salesOrder->accountingTransaction->total,
            'is_tax_included' => $salesOrder->accountingTransaction->is_tax_included,
            'link_id' => $salesCredit->id,
            'link_type' => SalesCredit::class,
            'type' => AccountingTransaction::TYPE_SALES_CREDIT,
            'reference' => $salesCredit->sales_credit_number,
        ]);

        AccountingTransactionLine::factory()->create([
            'accounting_transaction_id' => $accountingTransaction->id,
            'link_id' => $salesCreditLine->id,
            'link_type' => SalesCreditLine::class,
            'type' => AccountingTransactionLine::TYPE_SALES_CREDIT_LINE,
            'quantity' => $salesCreditLine->quantity,
            'amount' => $salesCreditLine->amount,
        ]);

        //Sales Credits
        if ($this->mockEnabled) {
            $this->clearExistingFakes();
            $this->mockUpdateOrCreateCreditMemos(SalesCredit::all());
            $this->mockCreateCustomer($salesCredit->customer);
        }

        $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
        Queue::pushed(QboUpdateOrCreateCreditMemosJob::class)->last()->handle();
        $this->assertEquals(1, QboCreditMemo::query()->count());
        $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());

        if (! $this->mockEnabled) {
            $qboCreditMemo = QboCreditMemo::query()->first();
            AccountingTransaction::where([
                'accounting_integration_id' => $qboCreditMemo->id,
                'accounting_integration_type' => QboCreditMemo::class,
            ])
                ->update([
                    'updated_at' => now()->addSecond(),
                ]);
            $this->postJson(route('qbo.transactions.syncTransactions', $this->qboIntegrationInstance->id))->assertOk();
            Queue::pushed(QboUpdateOrCreateCreditMemosJob::class)[1]->handle();
            $this->assertEquals(1, QboCreditMemo::where('updated_at', '>', $qboCreditMemo->updated_at)->count());
            $this->assertEquals(0, AccountingTransaction::whereNull('accounting_integration_id')->whereNull('accounting_integration_type')->count());
        }

        /*
        |--------------------------------------------------------------------------
        | Delete Payments
        |--------------------------------------------------------------------------
        */
        $this->postJson(route('qbo.transactions.deletePayments', $this->qboIntegrationInstance->id))->assertOk();

        if ($this->mockEnabled) {
            $this->clearExistingFakes();
            $this->mockDeletePayments(QboPayment::all());
        }

        Queue::pushed(QboDeletePaymentsJob::class)->last()->handle();
        $this->assertDatabaseEmpty(QboPayment::class);
    }
}
